/*
 *  Copyright 2013, Enguerrand de Rochefort
 * 
 * This file is part of xdat.
 *
 * xdat is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * xdat is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with xdat.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package workerThreads;

import gui.frames.ChartFrame;
import javax.swing.ProgressMonitor;
import javax.swing.SwingWorker;
import chart.Chart;
import xdat.Main;

/**
 * A thread that runs in the background to create a new chart.
 * This takes away this potentially long-running task from the EDT.
 * <br>
 * At the same time a ProgressMonitor is used to show progress, if required.
 * 
 */
public class ChartCreationThread 
extends SwingWorker 
{
	/** Flag to enable debug message printing for this class. */
	static final boolean printLog=false;
	
	/** The main window. */
	private Main mainWindow;
	
	/** The progress monitor. */
	private ProgressMonitor progressMonitor;
	
	/**
	 * Instantiates a new chart creation thread.
	 *
	 * @param mainWindow The main window
	 * @param progressMonitor The progress monitor
	 */
	public ChartCreationThread(Main mainWindow, ProgressMonitor progressMonitor) 
	{
		this.mainWindow = mainWindow;
		this.progressMonitor = progressMonitor;
	}

	@Override
	public Object doInBackground() 
	{
//		log("do in background invoked from Thread "+Thread.currentThread().getId());
//		log("Create: Creation invoked at "+(new Date()).toString());
		Chart chart = new Chart(mainWindow.getDataSheet(), progressMonitor);

//		log("doInBackground: progressMonitor isCanceled = "+progressMonitor.isCanceled());
		if(progressMonitor.isCanceled())
		{
//			log("doInBackground: progressMonitor is Canceled ");
			return null;
		}
		else
		{
//			log("Create: chart created. Creating ChartFrame. "+(new Date()).toString());
			new ChartFrame(mainWindow, chart);
//			log("Create: ChartFrame created. Adding chart to mainWindow.  "+(new Date()).toString());
			this.mainWindow.getCurrentSession().addChart(chart);
			return null;		
		}
	}


	/* (non-Javadoc)
	 * @see javax.swing.SwingWorker#done()
	 */
	@Override
	protected void done() {
		super.done();
		log("done: progressMonitor isCanceled = "+progressMonitor.isCanceled());
	}

	/**
	 * Prints debug information to stdout when printLog is set to true.
	 *
	 * @param message the message
	 */
	private void log(String message)
	{
		if(ChartCreationThread.printLog && Main.isLoggingEnabled())
		{
			System.out.println(this.getClass().getName()+"."+message);
		}
	}
}
