/*
 *  Copyright 2013, Enguerrand de Rochefort
 * 
 * This file is part of xdat.
 *
 * xdat is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * xdat is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with xdat.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package gui.dialogs;

import gui.WindowClosingAdapter;
import gui.frames.ChartFrame;
import gui.panels.AxisDisplaySettingsPanel;
import gui.panels.ChartDisplaySettingsPanel;

import java.awt.*;
import javax.swing.*;

import xdat.Main;

import chart.Chart;

import actionListeners.*;

/**
 * Dialog to modify display settings for a {@link chart.Chart}s, its axes or the {@link xdat.UserPreferences} 
 * for these display settings
 */
public class DisplaySettingsDialog 
extends JDialog 
{
	
	/** The version tracking unique identifier for Serialization. */
	static final long serialVersionUID = 0001;
	
	/** Flag to enable debug message printing for this class. */
	private static final boolean printLog=false;
	
	/** The main window. */
	private Main mainWindow;
	
	/** The chart display settings panel. */
	private ChartDisplaySettingsPanel chartDisplaySettingsPanel;
	
	/** The axis display settings panel. */
	private AxisDisplaySettingsPanel axisDisplaySettingsPanel;

	/**
	 * Instantiates a new display settings dialog.
	 *
	 * @param mainWindow the main window
	 * @throws HeadlessException the headless exception
	 */
	public DisplaySettingsDialog(Main mainWindow)
	throws HeadlessException 
	{
		super(mainWindow, "Default Display Settings", true);
		this.mainWindow = mainWindow;

		chartDisplaySettingsPanel = new ChartDisplaySettingsPanel(this.mainWindow, this ); 
		axisDisplaySettingsPanel = new AxisDisplaySettingsPanel(this.mainWindow, this); 
		chartDisplaySettingsPanel.setActionListener(new ChartDisplaySettingsActionListener(mainWindow, chartDisplaySettingsPanel));
		axisDisplaySettingsPanel.setActionListener(new AxisDisplaySettingsActionListener(mainWindow,this, axisDisplaySettingsPanel));
		buildDialog();
		
		chartDisplaySettingsPanel.setOkCancelButtonTargetDefaultSettings();
		axisDisplaySettingsPanel.setOkCancelButtonTargetDefaultSettings();

		this.setVisible(true);
	}

	/**
	 * Instantiates a new display settings dialog.
	 *
	 * @param mainWindow the main window
	 * @param chart the chart
	 * @param chartFrame the chart frame
	 * @throws HeadlessException the headless exception
	 */
	public DisplaySettingsDialog(Main mainWindow, Chart chart, ChartFrame chartFrame)
	throws HeadlessException 
	{
		super(chartFrame, "Display Settings");
		this.setModal(true);
		this.mainWindow = mainWindow;
		

		chartDisplaySettingsPanel = new ChartDisplaySettingsPanel(this.mainWindow, this, chartFrame ); 
		axisDisplaySettingsPanel = new AxisDisplaySettingsPanel(this.mainWindow, this, chartFrame); 
		chartDisplaySettingsPanel.setActionListener(new ChartDisplaySettingsActionListener(mainWindow, chartDisplaySettingsPanel, chart));
		axisDisplaySettingsPanel.setActionListener(new AxisDisplaySettingsActionListener(this, axisDisplaySettingsPanel, chart));

		buildDialog();
		
		chartDisplaySettingsPanel.setOkCancelButtonTargetChart(chart);
		axisDisplaySettingsPanel.setOkCancelButtonTargetChart(chart);

		log("preferred size : "+this.getPreferredSize().width+", "+ this.getPreferredSize().getHeight());
		this.setVisible(true);
		
	}
	
	/**
	 * Builds the dialog.
	 */
	private void buildDialog() {
		log("constructor called");
		this.addWindowListener(new WindowClosingAdapter(false));
		this.setResizable(false);
		
		// create components
		JTabbedPane tabbedPane = new JTabbedPane();

		//set Layouts
		this.setLayout(new BorderLayout());
				
		// add components
		this.add(tabbedPane, BorderLayout.CENTER);
		
		tabbedPane.add("General", chartDisplaySettingsPanel);
		tabbedPane.add("Axis-specific", axisDisplaySettingsPanel);
		//pack
		this.pack();

		//set location and make visible
		int left = (int)(0.5*this.mainWindow.getSize().width)- (int)(this.getSize().width*0.5) + this.mainWindow.getLocation().x;
		int top = (int)(0.5*this.mainWindow.getSize().height)- (int)(this.getSize().height*0.5) + this.mainWindow.getLocation().y;
		this.setLocation(left, top);
	}


	/**
	 * Gets the axis display settings panel.
	 *
	 * @return the axis display settings panel
	 */
	public AxisDisplaySettingsPanel getAxisDisplaySettingsPanel() {
		return axisDisplaySettingsPanel;
	}


	/**
	 * Gets the chart display settings panel.
	 *
	 * @return the chart display settings panel
	 */
	public ChartDisplaySettingsPanel getChartDisplaySettingsPanel() {
		return chartDisplaySettingsPanel;
	}


	/**
	 * Prints debug information to stdout when printLog is set to true.
	 *
	 * @param message the message
	 */
	private static final void log(String message)
	{
		if(DisplaySettingsDialog.printLog && Main.isLoggingEnabled())
		{
			System.out.println("DisplaySettingsDialog."+message);
		}
	}
	

}
