package xdat;
/*
 *  Copyright 2013, Enguerrand de Rochefort
 * 
 * This file is part of xdat.
 *
 * xdat is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * xdat is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with xdat.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

import gui.WindowClosingAdapter;
import gui.dialogs.LicenseDisplayDialog;
import gui.frames.ChartFrame;
import gui.menus.MainMenuBar;
import gui.panels.DataSheetTablePanel;

import java.awt.Dimension;
import java.io.IOException;
import java.io.InvalidClassException;
import java.util.Vector;
import java.util.regex.Pattern;

import javax.swing.*;
import chart.Chart;
import data.DataSheet;
/**
 * The Main Class from which the program is started. 
 * <p> Is also used to store some global references that are needed by other classes, such as for #
 * example references to Swing components.
 * Most of the data is stored in the {@link Session} and the {@link UserPreferences} classes though.
 * References to instances of both classes are kept in this class.
 */
public class Main 
extends JFrame
{
	
	/** The version tracking unique identifier for Serialization. */
	public static final long serialVersionUID = 0005;
	
	/** The release number used in the help -> about dialog. */
	public static final String versionString = "1.7";
	
	/** Flag to enable debug message printing with the log method for all classes. */
	public static final boolean loggingEnabled=false;

	/** Flag to enable debug message printing for this class. */
	public static final boolean printLog=false;
	
	/** The main menu bar. */
	private MainMenuBar mainMenuBar;
	
	/** The panel that contains the data. */
	private transient DataSheetTablePanel dataSheetTablePanel;
	
	/** The current session containing all relevant info in the memory. 
	 * This is also the information that gets serialized when saving a session. */
	private Session currentSession;
	
	/** A reference to all active chart frames. */
	private Vector<ChartFrame> chartFrames = new Vector<ChartFrame>(0,1);
	
	/**
	 * Instantiates a new main.
	 */
	public Main() 
	{
		super("xdat   -   Untitled");
		 new UserPreferences(versionString);
//		this.userPreferences.setLicenseAccepted(false);
		if(!this.checkLicense())
		{
			this.dispose();
			return;
		}
		this.currentSession = new Session();
		addWindowListener(new WindowClosingAdapter(true));
		this.mainMenuBar = new MainMenuBar(this);
		this.setJMenuBar(this.mainMenuBar);
		Dimension screenSize = getToolkit().getScreenSize();
		setLocation((int)(0.25*screenSize.width), (int)(0.25*screenSize.height));
		setSize((int)(0.5*screenSize.width), (int)(0.5*screenSize.height));
		setVisible(true);

	}

	/**
	 * To be called when the data in the panel has changed to update the GUI.
	 */
	public  void updateDataPanel()
	{
		log("updateDataPanel called");
		if(this.currentSession.getCurrentDataSheet() != null)
		{
			log("updateDataPanel: DataSheet non-null");
			this.dataSheetTablePanel = new DataSheetTablePanel(this, this.currentSession.getCurrentDataSheet());
			this.setContentPane(this.dataSheetTablePanel);
//			this.repaint();
			this.dataSheetTablePanel.revalidate();
		}
		else
		{
			this.setContentPane(new JPanel());
			this.repaint();
		}
	}
	
	/**
	 * The main method.
	 *
	 * @param args the command line arguments (not used)
	 */
	public static void main(String[] args)
	{		
		new Main();
	}

	/**
	 * Gets the data sheet.
	 *
	 * @return the data sheet
	 */
	public DataSheet getDataSheet() {
		return this.currentSession.getCurrentDataSheet();
	}

	/**
	 * Sets the data sheet.
	 *
	 * @param dataSheet the new data sheet
	 */
	public void setDataSheet(DataSheet dataSheet) {
		this.currentSession.setCurrentDataSheet(dataSheet);
		if(dataSheet == null)
		{
			this.remove(this.dataSheetTablePanel);
		}
		this.updateDataPanel();
		this.repaint();
	}

	/**
	 * Gets the user preferences.
	 *
	 * @return the user preferences
	 */
	public static UserPreferences getUserPreferences() {
		return new UserPreferences(versionString);
	}

	/**
	 * Gets the version string to be shown in the help->about dialog.
	 *
	 * @return the version string
	 */
	public static String getVersionString() {
		return versionString;
	}

	/**
	 * Gets the current session.
	 *
	 * @return the current session
	 */
	public Session getCurrentSession() {
		return currentSession;
	}

	/**
	 * Sets the current session.
	 *
	 * @param currentSession the new current session
	 */
	public void setCurrentSession(Session currentSession) {
		this.currentSession = currentSession;
	}	

	/**
	 * Adds a chart frame to the Vector with references to all chart frames.
	 *
	 * @param chartFrame the chart frame
	 */
	public void addChartFrame(ChartFrame chartFrame)
	{
		this.chartFrames.add(chartFrame);
	}
	
	/**
	 * Removes a chart frame from the Vector with references to all chart frames.
	 *
	 * @param chartFrame the chart frame
	 */
	public void removeChartFrame(ChartFrame chartFrame)
	{
		this.chartFrames.remove(chartFrame);
		this.currentSession.removeChart(chartFrame.getChart());
	}
	
	/**
	 * Gets a chart frame to the Vector with references to all chart frames.
	 *
	 * @param index the index
	 * @return the chart frame
	 */
	public ChartFrame getChartFrame(int index)
	{
		return this.chartFrames.get(index);
	}
	
	/**
	 * Gets the chart frame count.
	 *
	 * @return the number of active chart frames
	 */
	public int getChartFrameCount()
	{
		return this.chartFrames.size();
	}

	/**
	 * Dispose all chart frames.
	 */
	public void disposeAllChartFrames()
	{
		for(int i=this.chartFrames.size()-1; i>= 0; i--)
		{
			this.chartFrames.get(i).dispose();
		}
		this.chartFrames.removeAllElements();
		this.currentSession.clearAllCharts();
	}
	
	/**
	 * Update all chart frames when the data has changed.
	 */
	public void updateAllChartFrames()
	{
		log("updateAllChartFrames:------------------------------------- ");
		Chart[] charts = new Chart[this.currentSession.getChartCount()];
		log("updateAllChartFrames: currently "+charts.length+" charts active. ");
		for(int i=0; i<charts.length; i++)
		{
			log("updateAllChartFrames: reading chart "+i);
			charts[i] = (this.currentSession.getChart(i));
		}		
		this.disposeAllChartFrames();
		log("updateAllChartFrames: still "+charts.length+" charts active. ");
		for(int i=0; i<charts.length; i++)
		{
			log("updateAllChartFrames: creating chart "+i);
			this.chartFrames.add(new ChartFrame(this,charts[i]));
			this.currentSession.addChart(charts[i]);
			charts[i].autofitAllAxes();
			charts[i].evaluateBoundsForAllDesigns();
			charts[i].applyAllFilters();
		}	
		log("updateAllChartFrames: done. Session has now "+this.currentSession.getChartCount()+" active charts.");

	}
	
	/**
	 * Repaint all chart frames when the data has changed.
	 */
	public void repaintAllChartFrames()
	{
		for(int i=0;i<this.chartFrames.size();i++)
		{
			this.chartFrames.get(i).repaint();
		}
	}	
	
	
	/**
	 * Repaint all chart frames when the data has changed.
	 * 
	 * @param columnIndex the index of the column for which to reapply all filters
	 */
	public void refilterAllChartFrames(int columnIndex)
	{
		for(int i=0;i<this.chartFrames.size();i++)
		{
			this.chartFrames.get(i).getChart().getAxis(columnIndex).applyFilters();
		}
	}	
	
	/**
	 * Autofit a specified axis for all chart frames .
	 * 
	 * @param axisIndex the index of the axis to autofit
	 */
	public void autofitAxisAllChartFrames(int axisIndex)
	{
		for(int i=0;i<this.chartFrames.size();i++)
		{
			this.chartFrames.get(i).getChart().getAxis(axisIndex).autofit();
		}
	}	
	
	/**
	 * Resets filters for a specified axis for all chart frames .
	 * 
	 * @param axisIndex the index of the axis to autofit
	 */
	public void resetFiltersOnAxisAllChartFrames(int axisIndex)
	{
		for(int i=0;i<this.chartFrames.size();i++)
		{
			this.chartFrames.get(i).getChart().getAxis(axisIndex).resetFilters();
		}
	}
	
	/**
	 * Load session.
	 *
	 * @param pathToFile the path to the session file
	 */
	public void loadSession(String pathToFile)
	{
		try {
			this.disposeAllChartFrames();		
			this.currentSession = Session.readFromFile(this, pathToFile);
			
			this.setTitle("xdat   -   "+pathToFile);
			
			log("loadSession called. "+this.getChartFrameCount()+" chart frames to load.");
			
			ChartFrame[] chartFrames = new ChartFrame[this.currentSession.getChartCount()];
			for(int i=0; i<chartFrames.length; i++)
			{
				new ChartFrame(this,this.currentSession.getChart(i));
			}
			if(this.currentSession.getCurrentDataSheet() != null)
			{
				this.getMainMenuBar().setItemsRequiringDataSheetEnabled(true);
			}
			else
			{
				this.updateDataPanel();
				this.getMainMenuBar().setItemsRequiringDataSheetEnabled(false);
			}
				
		} catch (InvalidClassException e) {
			JOptionPane.showMessageDialog(this, "The file "+pathToFile+" is not a proper xdat version "+Main.versionString+" Session file", "Load Session", JOptionPane.OK_OPTION);
		} catch (ClassNotFoundException e) {
			JOptionPane.showMessageDialog(this, "The file "+pathToFile+" is not a proper xdat version "+Main.versionString+" Session file", "Load Session", JOptionPane.OK_OPTION);
		} catch (IOException e) {
			JOptionPane.showMessageDialog(this, "Error on loading session: "+ e.getMessage(), "Load Session", JOptionPane.OK_OPTION);
//			e.printStackTrace();
		}
		
	
		
	}

	/**
	 * Save session.
	 *
	 * @param pathToFile the path where the session should be saved.
	 */
	public void saveSessionAs(String pathToFile)
	{	
		for(int i=0; i<this.currentSession.getChartCount(); i++)
		{
			log("saving session with chart "+i);			
		}			
		try {
			this.currentSession.saveToFile(pathToFile);
		} catch (IOException e) {
//			JOptionPane.showMessageDialog(this, "IOException on saving session: " +e.getMessage(), "Save Session", JOptionPane.OK_OPTION);
			e.printStackTrace();
		}
	}

	private boolean checkLicense()
	{	
		if(!Main.getUserPreferences().isLicenseAccepted())
		{
			new LicenseDisplayDialog(Main.getUserPreferences()); 
		}
		
		return Main.getUserPreferences().isLicenseAccepted();
	}
	
	/**
	 * Checks if is debug message printing is enabled.
	 *
	 * @return true, if debug message printing is enabled
	 */
	public static boolean isLoggingEnabled() {
		return loggingEnabled;
	}

	/**
	 * Gets the main menu bar.
	 *
	 * @return the main menu bar
	 */
	public MainMenuBar getMainMenuBar() {
		return mainMenuBar;
	}

	
	/**
	 * Prints debug information to stdout when printLog is set to true.
	 *
	 * @param message the message
	 */
	private void log(String message)
	{
		if(Main.printLog && Main.isLoggingEnabled())
		{
			System.out.println(this.getClass().getName()+"."+message);
		}
	}

	/**
	 * Gets the Data Sheet Table Panel.
	 *
	 * @return  the Data Sheet Table Panel
	 */
	public DataSheetTablePanel getDataSheetTablePanel() {
		return dataSheetTablePanel;
	}
	
	
	
}

	

