/*
 *  Copyright 2011, Enguerrand de Rochefort
 * 
 * This file is part of xdat.
 *
 * xdat is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * xdat is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with xdat.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package gui.frames;

import gui.ChartFrameClosingAdapter;
import gui.menus.ChartFrameAddDesignToClusterMenu;
import gui.menus.ChartFrameMenuBar;
import gui.menus.ChartFrameSelectParametersMenu;
import gui.panels.ChartPanel;

import java.awt.GridLayout;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;

import javax.swing.JFrame;
import javax.swing.JScrollPane;

import xdat.Main;
import chart.Chart;


/**
 * A frame that us used to display a {@link chart.Chart}.
 */
public class ChartFrame 
extends JFrame 
implements ComponentListener
{
	
	/** The version tracking unique identifier for Serialization. */
	static final long serialVersionUID = 0001;
	
	/** Flag to enable debug message printing for this class. */
	static final boolean printLog=false;
	
	/** The main window. */
	private Main mainWindow;
	
	/** The chart. */
	private Chart chart;
	
	/** The panel on which the chart is painted. */
	private ChartPanel chartPanel;
	
	/** The {@link data.Cluster} menu. */
	private ChartFrameAddDesignToClusterMenu clusterMenu;
	
	/** The select {@link data.Parameter}s menu. */
	private ChartFrameSelectParametersMenu selectParametersMenu;
	
	/**
	 * Instantiates a new chart frame.
	 *
	 * @param mainWindow the main window
	 * @param chart the chart
	 */
	public ChartFrame(Main mainWindow, Chart chart)
	{
		super("Chart");
		log("constructor invoked.");
		addWindowListener(new ChartFrameClosingAdapter(this, mainWindow));
		this.mainWindow = mainWindow;

		this.chart = chart;
		mainWindow.addChartFrame(this);
		this.setJMenuBar(new ChartFrameMenuBar(mainWindow, this, chart));
		this.setLayout(new GridLayout(1,1));
		
		this.chartPanel = new ChartPanel(mainWindow.getDataSheet(), this.chart);
		JScrollPane scrollPane = new JScrollPane(this.chartPanel, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
		this.add(scrollPane);
		setLocation(this.chart.getLocation());
		setSize(this.chart.getSize());
		this.addComponentListener(this);
		setVisible(true);
	}
	
	/**
	 * Gets the chart.
	 *
	 * @return the chart
	 */
	public Chart getChart() {
		return chart;
	}
	
	/**
	 * Gets the chart panel.
	 *
	 * @return the chart panel
	 */
	public ChartPanel getChartPanel() {
		return chartPanel;
	}
	
	/* (non-Javadoc)
	 * @see java.awt.Window#dispose()
	 */
	public void dispose()
	{
		log("dispose called");
		mainWindow.removeChartFrame(this);
		super.dispose();
	}
	
	/**
	 * Prints debug information to stdout when printLog is set to true.
	 *
	 * @param message the message
	 */
	private void log(String message)
	{
		if(ChartFrame.printLog && Main.isLoggingEnabled())
		{
			System.out.println(this.getClass().getName()+"."+message);
		}
	}
	
	/**
	 * Gets the cluster menu.
	 *
	 * @return the cluster menu
	 */
	public ChartFrameAddDesignToClusterMenu getClusterMenu() {
		return clusterMenu;
	}
	
	/**
	 * Sets the cluster menu.
	 *
	 * @param clusterMenu the new cluster menu
	 */
	public void setClusterMenu(ChartFrameAddDesignToClusterMenu clusterMenu) {
		this.clusterMenu = clusterMenu;
	}
	
	/**
	 * Gets the select parameters menu.
	 *
	 * @return the select parameters menu
	 */
	public ChartFrameSelectParametersMenu getSelectParametersMenu() {
		return selectParametersMenu;
	}
	
	/**
	 * Sets the select parameters menu.
	 *
	 * @param selectParametersMenu the new select parameters menu
	 */
	public void setSelectParametersMenu(
			ChartFrameSelectParametersMenu selectParametersMenu) {
		this.selectParametersMenu = selectParametersMenu;
	}
	
	/* (non-Javadoc)
	 * @see java.awt.event.ComponentListener#componentHidden(java.awt.event.ComponentEvent)
	 */
	public void componentHidden(ComponentEvent arg0) 
	{
	}
	
	/* (non-Javadoc)
	 * @see java.awt.event.ComponentListener#componentMoved(java.awt.event.ComponentEvent)
	 */
	public void componentMoved(ComponentEvent arg0) 
	{
		this.chart.setLocation(this.getLocation());
	}
	
	/* (non-Javadoc)
	 * @see java.awt.event.ComponentListener#componentResized(java.awt.event.ComponentEvent)
	 */
	public void componentResized(ComponentEvent arg0) 
	{
		this.chart.setSize(this.getSize());
	}
	
	/* (non-Javadoc)
	 * @see java.awt.event.ComponentListener#componentShown(java.awt.event.ComponentEvent)
	 */
	public void componentShown(ComponentEvent arg0) 
	{
	}
}
