/*
 *  Copyright 2011, Enguerrand de Rochefort
 * 
 * This file is part of xdat.
 *
 * xdat is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * xdat is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with xdat.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package chart;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Point;
import java.io.Serializable;
import java.util.*;
import javax.swing.ProgressMonitor;

import xdat.*;
import data.*;



/**
 * A serializable representation of all relevant settings for a Chart which is displayed on 
 * a ChartFrame.
 * <p>
 * The data that was imported by the user can be displayed on a Chart. 
 * The Chart is built by as many vertical Axes as Parameters are present in the underlying the DataSheet and
 * each Axis represents one of these Parameters.
 * <p>
 * The Designs (or, in other words, rows of the data table) are represented by lines that connect points
 * on the Axes. Each Design's line crosses each Axis exactly at the ordinate that corresponds to the
 * value for the respective Parameter in the Design. This allows to display the whole DataSheet in just
 * one Chart, irrespective of how many dimensions it has.
 * <p>
 * The Chart also provides interactivity through a pair of draggable Filters that are present on 
 * each Axis. Depending on the positions of these filters, certain Designs are filtered from the 
 * display. This means that they are either displayed in a different color or hidden completely.
 * 
 * 
 * @see gui.frames.ChartFrame
 * @see Axis
 * @see Filter
 * @see data.Parameter
 * @see data.DataSheet
 * @see data.Design
 */
public class Chart 
implements Serializable
{
	
	/** The version tracking unique identifier for Serialization. */
	static final long serialVersionUID = 3;
	
	/** Flag to enable debug message printing for this class. */
	static final boolean printLog=false;
	
	/** The location of the {@link gui.frames.ChartFrame} on the screen. */
	private Point location;
	
	/** The size of this Chart. */
	private Dimension size;
	
	/** The data sheet that is displayed in this Chart.
	 * 
	 * @see data.DataSheet */
	private DataSheet dataSheet;
	
	/** The top margin of the Chart.
	 * 
	 *  specifies the distance from the top edge of the Chart to the top of the Axis labels. */
	private int topMargin = 10;
	
	/** The axes on this Chart. 
	 * <p>
	 * For each Parameter in the DataSheet there is exactly one Axis.
	 * */
	private Vector<Axis> axes = new Vector<Axis>(0,1);
	
	/** The background color of this Chart. */
	private Color backGroundColor;
	
	/** The design label font size.
	 * <p>
	 * The design IDs are shown as labels next to the left-most Axis. This field specifies the font size for these labels. */
	private int designLabelFontSize;
	
	/** The active design color. 
	 * <p>
	 * All Designs that are not filtered and do not belong to any Clusters are displayed in this Color.
	 * New Clusters are also given this Color by default.
	 * 
	 * @see data.Design
	 * @see data.Cluster 
	 * */
	private Color activeDesignColor;
	
	/** The filtered design color. 
	 * <p>
	 * All Designs that are filtered are displayed in this Color.
	 * This is only relevant if {@link #showFilteredDesigns} is true.
	 * 
	 * @see data.Design
	 * */
	private Color filteredDesignColor;
	
	/** The color in which the Filters are shown on this Chart. */
	private Color filterColor;
	
	/** Specifies whether the design IDs next to the left-most Axis should be shown. */
	private boolean showDesignIDs;
	
	/** Switch that enables displaying filtered designs. 
	 * <p>
	 * If this switch is true, designs are displayed in the Color specified by {@link #filteredDesignColor}*/
	private boolean showFilteredDesigns;
	
	/** The height of the triangles that represent the filter in pixels. */
	private int filterHeight;
	
	/** The width of one half triangle that represents a filter in pixels. In other words, 
	 * the filter triangle will be twice as large as the value entered here. */
	private int filterWidth;
	
	/**
	 * Instantiates a new chart.
	 *
	 * @param dataSheet the data sheet
	 */
	public Chart(DataSheet dataSheet, ProgressMonitor progressMonitor)
	{
		this.location = new Point(100,100);
		this.size = new Dimension(800,600);
		log("constructor called. Read Base settings.");
		this.dataSheet = dataSheet;
		UserPreferences userPreferences = Main.getUserPreferences();
		this.backGroundColor =userPreferences.getDefaultBackgroundColor();
		this.showDesignIDs = userPreferences.isShowDesignIDs();
		this.showFilteredDesigns = userPreferences.isShowFilteredDesigns();
		this.activeDesignColor = userPreferences.getActiveDesignDefaultColor();
		this.filteredDesignColor = userPreferences.getFilteredDesignDefaultColor();
		this.designLabelFontSize = userPreferences.getDesignLabelFontSize();
		this.filterColor = userPreferences.getFilterDefaultColor();
		this.filterHeight = userPreferences.getFilterHeight();
		this.filterWidth=userPreferences.getFilterWidth();
		log("constructor: Base settings read. Creating axes...");
		progressMonitor.setMaximum(dataSheet.getParameterCount()-1);
		progressMonitor.setNote("Building Chart...");
		for (int i=0; i<dataSheet.getParameterCount() && !progressMonitor.isCanceled(); i++)
		{
			log("constructor: Creating axis "+dataSheet.getParameter(i).getName());
			log("constructor: progress monitor isCanceled(): "+progressMonitor.isCanceled());
			Axis newAxis = new Axis(dataSheet, this, dataSheet.getParameter(i));
			this.addAxis(newAxis);
			progressMonitor.setProgress(i);
		}

		if(!progressMonitor.isCanceled())
		{
			log("constructor: progress monitor isCanceled(): "+progressMonitor.isCanceled());
			progressMonitor.setNote("Building Filters...");
			progressMonitor.setProgress(0);
			log("constructor: axes created. Creating filters...");
			log("constructor: progress monitor isCanceled(): "+progressMonitor.isCanceled());
			for (int i=0; i<dataSheet.getParameterCount() && !progressMonitor.isCanceled(); i++)
			{
				this.axes.get(i).addFilters();
				progressMonitor.setProgress(i);
			}
			log("constructor: filters created. ");		
			log("constructor: progress monitor isCanceled(): "+progressMonitor.isCanceled());
		}
	}
	
	/**
	 * Determines the width of this Chart.
	 *
	 * @return the width of this Chart
	 */
	public int getWidth()
	{
		int width = 0;
		if(this.getAxis(0).isActive())
		{
			width = width + (int)(0.5 * this.getAxis(0).getWidth());
		}	
		for(int i=1; i<this.getAxisCount(); i++)
		{
			if(this.getAxis(i).isActive())
			{
				width = width + (int)(this.getAxis(i).getWidth());
			}
		}
		return width;
	}
	
	/**
	 * Determines the height of this Chart.
	 *
	 * @return the height of this Chart
	 */
	public int getHeight()
	{
		int height = getAxisTopPos()+getAxisMaxHeight();
		return height;
	}

	/**
	 * Gets the largest height of all Axis heights.
	 *
	 * @return the largest height of all Axis heights
	 */
	public int getAxisMaxHeight()
	{
		int height = 0;
		for(int i=0; i<this.getAxisCount(); i++)
		{
			if(this.getAxis(i).isActive())
			{
				if(height < this.getAxis(i).getHeight())
				{
					height = this.getAxis(i).getHeight();
				}
			}
		}
		return height;
	}	
		
	/**
	 * Sets the height of this Chart by setting the heigh of all Axes.
	 *
	 * @param height the height of this Chart by setting the heigh of all Axes..
	 */
	public void setAxisHeight(int height)
	{
		for (int i=0; i<axes.size(); i++)
		{
			axes.get(i).setHeight(height);
		}
	}
	
	/**
	 * Gets the largest width of all Axis widths.
	 *
	 * @return the largest width of all Axis widths.
	 */
	public int getAxisMaxWidth()
	{
		int width = 0;
		for(int i=0; i<this.getAxisCount(); i++)
		{
			if(this.getAxis(i).isActive())
			{
				if(width < this.getAxis(i).getWidth())
				{
					width = this.getAxis(i).getWidth();
				}
			}
		}
		return width;
	}	
		
	/**
	 * Sets the axis width.
	 *
	 * @param width the new axis width
	 */
	public void setAxisWidth(int width)
	{
		for (int i=0; i<axes.size(); i++)
		{
			axes.get(i).setWidth(width);
		}
	}
	
	/**
	 * Sets the axis color.
	 *
	 * @param color the new axis color
	 */
	public void setAxisColor(Color color)
	{
		for (int i=0; i<axes.size(); i++)
		{
			axes.get(i).setAxisColor(color);
		}
	}
	
	/**
	 * Gets the position in pixels of the top of the Axes of this Chart.
	 *
	 * @return the position in pixels of the top of the Axes of this Chart.
	 */
	public int getAxisTopPos()
	{
		log("getAxisTopPos: returning "+(getMaxAxisLabelFontSize()+this.getTopMargin()*2+this.getFilterHeight()));
		return getMaxAxisLabelFontSize()+this.getTopMargin()*2+this.getFilterHeight();
	}
	
	/**
	 * Gets an Axis by its index.
	 *
	 * @param index the index
	 * @return the Axis with index index
	 */
	public Axis getAxis(int index) {
		
		return axes.get(index);
	}
	
	/**
	 * Gets an Axis by its name.
	 *
	 * @param parameterName the parameter name
	 * @return the Axis
	 */
	public Axis getAxis(String parameterName) 
	{
		for(int i=0; i<this.axes.size(); i++)
		{
			if(parameterName.equals(this.axes.get(i).getParameter().getName()))
			{
				return this.axes.get(i);
			}
		}
		throw new IllegalArgumentException("Axis "+parameterName+" not found");	
	}
	
	/**
	 * Gets the largest Axis label font size on this Chart.
	 *
	 * @return the largest axis label font size on this Chart
	 */
	public int getMaxAxisLabelFontSize() 
	{
		int maxAxisLabelFontSize = 0;
		for (int i=0; i<axes.size(); i++)
		{
			if(maxAxisLabelFontSize<axes.get(i).getAxisLabelFontSize())
			{
				maxAxisLabelFontSize = axes.get(i).getAxisLabelFontSize();
			}
		}
		return maxAxisLabelFontSize;
	}

	/**
	 * Gets the active axis count.
	 *
	 * @return the active axis count
	 */
	public int getActiveAxisCount() {
		int axesCount=0;
		for(int i=0; i<axes.size(); i++)
		{
			if(axes.get(i).isActive())
				axesCount++;
		}
		return axesCount;
	}
	
	/**
	 * Gets the axis count.
	 *
	 * @return the axis count
	 */
	public int getAxisCount() {
		return axes.size();
	}
	
	/**
	 * Adds the axis.
	 *
	 * @param axis the axis
	 */
	public void addAxis(Axis axis) 
	{
		this.axes.add(axis);
	}
	
	/**
	 * Adds an Axis at the position index.
	 *
	 * @param index the index where the Axis should be added
	 * @param axis the Axis to be added
	 */
	public void addAxis(int index, Axis axis) 
	{
		this.axes.add(index, axis);
	}

	/**
	 * Removes the axis with index index.
	 *
	 * @param index the index of the Axis to be removed
	 */
	public void removeAxis(int index) 
	{
		this.axes.remove(index);
	}

	/**
	 * Removes the axis with name name.
	 *
	 * @param parameterName the name of the parameter for which the Axis should be removed
	 */
	public void removeAxis(String parameterName) 
	{
		for(int i=0; i<this.axes.size(); i++)
		{
			if(parameterName.equals(this.axes.get(i).getParameter().getName()))
			{
				this.axes.remove(i);
				return;
			}
		}
		throw new IllegalArgumentException("Axis "+parameterName+" not found");	
	}
	
	/**
	 * Function to reorder the axes in the chart
	 *
	 * @param oldIndex the index of the axis to be moved
	 * @param newIndex the target index for the axis  to be moved
	 */	
	public void moveAxis(int oldIndex, int newIndex)
	{
		log("moveAxis called with arguments "+oldIndex+" and "+newIndex);
		Axis axis= this.axes.remove(oldIndex);
		this.axes.insertElementAt(axis, newIndex);
	}	
	
	
	/**
	 * Gets the data sheet.
	 *
	 * @return the data sheet
	 */
	public DataSheet getDataSheet() {
		return dataSheet;
	}
	
	/**
	 * Sets the data sheet.
	 *
	 * @param dataSheet the new data sheet
	 */
	public void setDataSheet(DataSheet dataSheet) {
		this.dataSheet = dataSheet;
	}
	
	/**
	 * Prints debug information to stdout when printLog is set to true.
	 *
	 * @param message the message
	 */
	private void log(String message)
	{
		if(Chart.printLog && Main.isLoggingEnabled())
		{
			System.out.println(this.getClass().getName()+"."+message);
		}
	}

	/**
	 * Gets the design label font size.
	 *
	 * @return the design label font size
	 */
	public int getDesignLabelFontSize() {
		return designLabelFontSize;
	}

	/**
	 * Sets the design label font size.
	 *
	 * @param designLabelFontSize the new design label font size
	 */
	public void setDesignLabelFontSize(int designLabelFontSize) {
		this.designLabelFontSize = designLabelFontSize;
	}

	/**
	 * Gets the design color.
	 *
	 * @param design the design
	 * @param designActive the design active
	 * @return the design color
	 */
	public Color getDesignColor(Design design, boolean designActive) 	// design active is function argument to improve performance
	{
		if(designActive && design.getCluster()!=null)
		{
			return design.getCluster().getActiveDesignColor();
		}
		else if(designActive)
		{
			return activeDesignColor;
		}
			
		else
		{
			return filteredDesignColor;
		}
	}

	/**
	 * Gets the default design color.
	 *
	 * @param designActive the design active
	 * @return the default design color
	 */
	public Color getDefaultDesignColor(boolean designActive) {
		if(designActive)
			return activeDesignColor;
		else
			return filteredDesignColor;
	}
	
	/**
	 * Sets the active design color.
	 *
	 * @param activeDesignColor the new active design color
	 */
	public void setActiveDesignColor(Color activeDesignColor) {
		this.activeDesignColor = activeDesignColor;
	}

	/**
	 * Sets the filtered design color.
	 *
	 * @param filteredDesignColor the new filtered design color
	 */
	public void setFilteredDesignColor(Color filteredDesignColor) {
		this.filteredDesignColor = filteredDesignColor;
	}

	/**
	 * Gets the back ground color.
	 *
	 * @return the back ground color
	 */
	public Color getBackGroundColor() {
		return backGroundColor;
	}

	/**
	 * Sets the back ground color.
	 *
	 * @param backGroundColor the new back ground color
	 */
	public void setBackGroundColor(Color backGroundColor) {
		this.backGroundColor = backGroundColor;
	}

	/**
	 * Checks if design IDs should be shown.
	 *
	 * @return true, if design IDs should be shown.
	 */
	public boolean isShowDesignIDs() {
		return showDesignIDs;
	}

	/**
	 * Specifies whether design IDs should be shown.
	 *
	 * @param showDesignIDs Specifies whether design IDs should be shown.
	 */
	public void setShowDesignIDs(boolean showDesignIDs) {
		this.showDesignIDs = showDesignIDs;
	}

	/**
	 * Checks whether filtered designs should be shown.
	 *
	 * @return true, if filtered designs should be shown.
	 */
	public boolean isShowFilteredDesigns() {
		return showFilteredDesigns;
	}

	/**
	 * Specifies whether filtered designs should be shown.
	 *
	 * @param showFilteredDesigns specifies whether filtered designs should be shown.
	 */
	public void setShowFilteredDesigns(boolean showFilteredDesigns) {
		this.showFilteredDesigns = showFilteredDesigns;
	}

	/**
	 * Gets the filter color.
	 *
	 * @return the filter color
	 */
	public Color getFilterColor() {
		return filterColor;
	}

	/**
	 * Sets the filter color.
	 *
	 * @param filterColor the new filter color
	 */
	public void setFilterColor(Color filterColor) {
		this.filterColor = filterColor;
	}

	/**
	 * Gets the top margin.
	 *
	 * @return the top margin
	 */
	public int getTopMargin() {
		return topMargin;
	}

	/**
	 * Reset display settings to default.
	 */
	public void resetDisplaySettingsToDefault()
	{
		UserPreferences userPreferences = Main.getUserPreferences();
		this.backGroundColor =userPreferences.getDefaultBackgroundColor();
		this.showDesignIDs = userPreferences.isShowDesignIDs();
		this.showFilteredDesigns = userPreferences.isShowFilteredDesigns();
		this.activeDesignColor = userPreferences.getActiveDesignDefaultColor();
		this.filteredDesignColor = userPreferences.getFilteredDesignDefaultColor();
		this.designLabelFontSize = userPreferences.getDesignLabelFontSize();
		this.filterColor = userPreferences.getFilterDefaultColor();	
		this.filterHeight = userPreferences.getFilterHeight();
		this.filterWidth=userPreferences.getFilterWidth();
		for (int i=0; i<axes.size(); i++)
		{
			axes.get(i).resetSettingsToDefault();
		}
	}

	/**
	 * Gets the filter height.
	 *
	 * @return the filter height
	 */
	public int getFilterHeight() {
		return filterHeight;
	}

	/**
	 * Sets the filter height.
	 *
	 * @param filterHeight the new filter height
	 */
	public void setFilterHeight(int filterHeight) {
		this.filterHeight = filterHeight;
	}

	/**
	 * Gets the filter width.
	 *
	 * @return the filter width
	 */
	public int getFilterWidth() {
		return filterWidth;
	}

	/**
	 * Sets the filter width.
	 *
	 * @param filterWidth the new filter width
	 */
	public void setFilterWidth(int filterWidth) {
		this.filterWidth = filterWidth;
	}

	/**
	 * Apply all filters.
	 */
	public void applyAllFilters()
	{
		for (int i=0; i<dataSheet.getParameterCount(); i++)
		{
			this.axes.get(i).applyFilters();
		}			
	}
	
	/**
	 *Autofits all axes.
	 */
	public void autofitAllAxes()
	{
		for (int i=0; i<this.axes.size(); i++)
		{
			this.axes.get(i).autofit();
		}			
	}
	
	/**
	 * Evaluate Axis bounds for all designs.
	 * @see DataSheet
	 */
	public void evaluateBoundsForAllDesigns()
	{
		this.dataSheet.evaluateBoundsForAllDesigns(this);
	}

	/**
	 * Gets the size of this Chart.
	 *
	 * @return the size of this Chart.
	 */
	public Dimension getSize() {
		return size;
	}

	/**
	 * Sets the size of this Chart..
	 *
	 * @param size the new size of this Chart.
	 */
	public void setSize(Dimension size) {
		this.size = size;
	}

	/**
	 * Gets the location of this Chart on the Screen.
	 *
	 * @return the location of this Chart on the Screen.
	 */
	public Point getLocation() 
	{
		return location;
	}

	/**
	 * Sets the location of this Chart on the Screen..
	 *
	 * @param location the new location of this Chart on the Screen.
	 */
	public void setLocation(Point location) 
	{
		this.location = location;
	}

	


	
}
