/*
 *  Copyright 2012, Enguerrand de Rochefort
 * 
 * This file is part of xdat.
 *
 * xdat is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * xdat is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with xdat.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */


package xdat;

import java.awt.Color;
import java.util.Locale;
import java.util.prefs.*;

/**
 * This class stores all user preference settings.
 * 
 * When the user makes changes to settings such as default background color of 
 * charts or delimiting characters for data imports these settings are stored 
 * using the java Preferences API. 
 */
public class UserPreferences 
{
	
	/** The version tracking unique identifier for Serialization. */
	static final long serialVersionUID = 0003;
	
	/** Flag to enable debug message printing for this class. */
	static final boolean printLog=false;
	
	/** Preferences Object to store settings. */
	private Preferences prefs;
	
	/** Current Release Version to be able to store version-specific boolean value for the click-wrap license.**/
	private String versionString;
	
	//File Options
	/** Open file import browse dialog in the user's home directory by default. */
	public static final int IMPORT_FROM_HOMEDIR = 0;

	/** Open file import browse dialog in the last opened directory by default.   */
	public static final int IMPORT_FROM_LASTDIR = 1;
	
	/** Open file import browse dialog in a userspecified directory by default.  */
	public static final int IMPORT_FROM_USERDIR = 2;
	
	//Number Formats
	/** US Locale for number formatting. */
	public static final int LOCALE_US = 0;

	/** German Locale for number formatting.   */
	public static final int LOCALE_DE = 1;

	/**
	 * Instantiates a new user preferences object.
	 */
	public UserPreferences(String versionString)
	{
		log("constructor called.");
		this.prefs = Preferences.userNodeForPackage(getClass());
		this.versionString = versionString;
	}

	/**
	 * Returns true, when the user has accepted the license
	 *
	 * @return whether the license was accepted by the user
	 */
	public boolean isLicenseAccepted()
	{
		return this.prefs.getBoolean("version"+this.versionString+"licenseAcceptedBy"+System.getProperty("user.name"), false);
	}

	
	/**
	 * Specifies whether the user has accepted the license.
	 *
	 * @param licenseAccepted the flag that specifies whether the user has accepted the license
	 */
	public void setLicenseAccepted(boolean licenseAccepted)
	{
		this.prefs.putBoolean("version"+this.versionString+"licenseAcceptedBy"+System.getProperty("user.name"), licenseAccepted);
	}	

	/**
	 * Gets the axis color
	 *
	 * @return the axis color
	 */
	public Color getAxisColor() 
	{
		int r=this.prefs.getInt("axisColorRed", 0);
		int g=this.prefs.getInt("axisColorGreen", 0);
		int b=this.prefs.getInt("axisColorBlue", 0);
		return new Color(r,g,b);
	}
	
	/**
	 * Sets the axis color.
	 *
	 * @param axisColor the new axis color
	 */
	public void setAxisColor(Color axisColor) {
		this.prefs.putInt("axisColorRed", axisColor.getRed());
		this.prefs.putInt("axisColorGreen", axisColor.getGreen());
		this.prefs.putInt("axisColorBlue", axisColor.getBlue());
	}
	
	/**
	 * Gets the axis height.
	 *
	 * @return the axis height
	 */
	public int getAxisHeight() 
	{
		return this.prefs.getInt("axisHeight",400);
	}
	
	/**
	 * Sets the axis height.
	 *
	 * @param axisHeight the new axis height
	 */
	public void setAxisHeight(int axisHeight) {
		this.prefs.putInt("axisHeight",axisHeight);
	}
	
	/**
	 * Gets the axis label font color.
	 *
	 * @return the axis label font color
	 */
	public Color getAxisLabelFontColor() {
		int r=this.prefs.getInt("axisLabelFontColorRed", 0);
		int g=this.prefs.getInt("axisLabelFontColorGreen", 0);
		int b=this.prefs.getInt("axisLabelFontColorBlue", 0);
		return new Color(r,g,b);
	}
	
	/**
	 * Sets the axis label font color.
	 *
	 * @param axisLabelFontColor the new axis label font color
	 */
	public void setAxisLabelFontColor(Color axisLabelFontColor) {
		this.prefs.putInt("axisLabelFontColorRed", axisLabelFontColor.getRed());
		this.prefs.putInt("axisLabelFontColorGreen", axisLabelFontColor.getGreen());
		this.prefs.putInt("axisLabelFontColorBlue", axisLabelFontColor.getBlue());
	}
	
	/**
	 * Gets the axis label font size.
	 *
	 * @return the axis label font size
	 */
	public int getAxisLabelFontSize() {
		return this.prefs.getInt("axisLabelFontSize",20);
	}
	
	/**
	 * Sets the axis label font size.
	 *
	 * @param axisLabelFontSize the new axis label font size
	 */
	public void setAxisLabelFontSize(int axisLabelFontSize) {
		this.prefs.putInt("axisLabelFontSize",axisLabelFontSize);
	}
	
	/**
	 * Gets the axis tick count.
	 *
	 * @return the axis tick count
	 */
	public int getAxisTicCount() {
		return this.prefs.getInt("axisTicCount",11);
	}
	
	/**
	 * Sets the axis tick count.
	 *
	 * @param axisTicCount the new axis tick count
	 */
	public void setAxisTicCount(int axisTicCount) {
		this.prefs.putInt("axisTicCount",axisTicCount);
	}
	
	/**
	 * Gets the axis tick label font color.
	 *
	 * @return the axis tick label font color
	 */
	public Color getAxisTicLabelFontColor() {
		int r=this.prefs.getInt("axisTicLabelFontColorRed", 0);
		int g=this.prefs.getInt("axisTicLabelFontColorGreen", 0);
		int b=this.prefs.getInt("axisTicLabelFontColorBlue", 0);
		return new Color(r,g,b);
	}
	
	/**
	 * Sets the axis tick label font color.
	 *
	 * @param axisTicLabelFontColor the new axis tick label font color
	 */
	public void setAxisTicLabelFontColor(Color axisTicLabelFontColor) {
		this.prefs.putInt("axisTicLabelFontColorRed", axisTicLabelFontColor.getRed());
		this.prefs.putInt("axisTicLabelFontColorGreen", axisTicLabelFontColor.getGreen());
		this.prefs.putInt("axisTicLabelFontColorBlue", axisTicLabelFontColor.getBlue());
	}
	
	/**
	 * Gets the axis tick label format.
	 *
	 * @return the axis tick label format
	 */
	public String getAxisTicLabelFormat() {
		return this.prefs.get("axisTicLabelFormat","%4.3f");
	}
	
	/**
	 * Sets the axis tick label format.
	 *
	 * @param axisTicLabelFormat the new axis tick label format
	 */
	public void setAxisTicLabelFormat(String axisTicLabelFormat) {
		this.prefs.put("axisTicLabelFormat",axisTicLabelFormat);
	}
	
	/**
	 * Gets the axis tick length.
	 *
	 * @return the axis tick length
	 */
	public int getAxisTicLength() {
		return this.prefs.getInt("axisTicLength",4);
	}
	
	/**
	 * Sets the axis tick length.
	 *
	 * @param axisTicLength the new axis tick length
	 */
	public void setAxisTicLength(int axisTicLength) {
		this.prefs.putInt("axisTicLength",axisTicLength);
	}
	
	/** Gets the width in pixels that is used by one axis. 
	 * This setting is used to define the axis spacing. The distance of two axes is defined by the sum of their respective widths, divided by two. 
	 * 
	 * @return the axis width
	 */
	public int getAxisWidth() {
		return this.prefs.getInt("axisWidth",200);
	}
	
	/**
	 * Sets the axis width.
	 *
	 * @param axisWidth the new axis width
	 */
	public void setAxisWidth(int axisWidth) {
		this.prefs.putInt("axisWidth",axisWidth);
	}
	
	/**
	 * Gets the filter default color.
	 *
	 * @return the filter default color
	 */
	public Color getFilterDefaultColor() {
		int r=this.prefs.getInt("filterColorRed", 255);
		int g=this.prefs.getInt("filterColorGreen", 0);
		int b=this.prefs.getInt("filterColorBlue", 0);
		return new Color(r,g,b);
	}
	
	/**
	 * Sets the filter color.
	 *
	 * @param filterColor the new filter color
	 */
	public void setFilterColor(Color filterColor) {
		this.prefs.putInt("filterColorRed", filterColor.getRed());
		this.prefs.putInt("filterColorGreen", filterColor.getGreen());
		this.prefs.putInt("filterColorBlue", filterColor.getBlue());
	}
	
	/**
	 * Gets the filter height.
	 *
	 * @return the filter height
	 */
	public int getFilterHeight() {
		return this.prefs.getInt("filterHeight",7);
	}
	
	/**
	 * Sets the filter height.
	 *
	 * @param filterHeight the new filter height
	 */
	public void setFilterHeight(int filterHeight) {
		this.prefs.putInt("filterHeight",filterHeight);
	}
	
	/** Gets the width of one half triangle that represents a filter in pixels. In other words, 
	 * the filter triangle will be twice as large as the value entered here. 
	 *
	 * @return the filter width
	 */
	public int getFilterWidth() {
		return this.prefs.getInt("filterWidth",5);
	}
	
	/**
	 * Sets the filter width.
	 *
	 * @param filterWidth the new filter width
	 */
	public void setFilterWidth(int filterWidth) {
		this.prefs.putInt("filterWidth",filterWidth);
	}
	
	/**
	 * Gets the axis tick label font size.
	 *
	 * @return the axis tick label font size
	 */
	public int getAxisTicLabelFontSize() {
		return this.prefs.getInt("ticLabelFontSize",10);
	}
	
	/**
	 * Sets the axis tick label font size.
	 *
	 * @param ticLabelFontSize the new axis tick label font size
	 */
	public void setAxisTicLabelFontSize(int ticLabelFontSize) {
		this.prefs.putInt("ticLabelFontSize",ticLabelFontSize);
	}
	
	/**
	 * Gets the design label font size.
	 *
	 * @return the design label font size
	 */
	public int getDesignLabelFontSize() {
		return this.prefs.getInt("designLabelFontSize",10);
	}
	
	/**
	 * Sets the design label font size.
	 *
	 * @param designLabelFontSize the new design label font size
	 */
	public void setDesignLabelFontSize(int designLabelFontSize) {
		this.prefs.putInt("designLabelFontSize",designLabelFontSize);
	}

	/**
	 * Gets the dir to import from.
	 *
	 * @return the dir to import from
	 */
	public int getDirToImportFrom() {
		return this.prefs.getInt("dirToImportFrom",IMPORT_FROM_LASTDIR);
	}
	
	/**
	 * Sets the dir to import from.
	 *
	 * @param dirToImportFrom the new dir to import from
	 */
	public void setDirToImportFrom(int dirToImportFrom) {
		this.prefs.putInt("dirToImportFrom",dirToImportFrom);
	}
	
	/**
	 * Gets the last file.
	 *
	 * @return the last file
	 */
	public String getLastFile() 
	{
		return this.prefs.get("lastFileBrowsingDirectory",System.getProperty("user.home"));
	}
	
	/**
	 * Sets the last file.
	 *
	 * @param lastFileBrowsingDirectory the new last file
	 */
	public void setLastFile(String lastFileBrowsingDirectory) {
		this.prefs.put("lastFileBrowsingDirectory",lastFileBrowsingDirectory);
	}

	/**
	 * Checks if the last file field has been initialised.
	 *
	 * @return true, if is last file field has been initialised
	 */
	public boolean isLastFileInitialised() 
	{
		return this.prefs.getBoolean("lastFileInitialised",false);
	}
	
	/**
	 * Gets the user home directory.
	 *
	 * @return the user home directory
	 */
	public String getHomeDir() {
		return this.prefs.get("homeDir",System.getProperty("user.home"));
	}
	
	/**
	 * Sets the home directory.
	 *
	 * @param homeDir the new home directory
	 */
	public void setHomeDir(String homeDir) {
		this.prefs.put("homeDir",homeDir);
	}
	
	/**
	 * Gets the user directory.
	 *
	 * @return the user directory
	 */
	public String getUserDir() {
		return this.prefs.get("userDir",System.getProperty("user.home"));
	}
	
	/**
	 * Sets the user directory.
	 *
	 * @param userDir the new user directory
	 */
	public void setUserDir(String userDir) {
		this.prefs.put("userDir",userDir);
	}
	
	/**
	 * Gets the directory for the file browsing dialog.
	 *
	 * @return the current directory based on the user preferences.
	 */
	public String getCurrentDir()
	{
		switch(this.getDirToImportFrom())
		{
			case(IMPORT_FROM_HOMEDIR):
			{
				return this.getHomeDir();
			}
			case(IMPORT_FROM_LASTDIR):
			{
				return this.getLastFile();
			}
			case(IMPORT_FROM_USERDIR):
			{
				return this.getUserDir();
			}
			default:
			{
				return this.getHomeDir();
			}
		}
		
	}
	
	/**
	 * Gets the delimiter for importing data.
	 *
	 * @return the delimiter
	 */
	public String getDelimiter() {
		return this.prefs.get("delimiter","\\s");
	}
	
	/**
	 * Sets the delimiter for importing data.
	 *
	 * @param delimiter the new delimiter
	 */
	public void setDelimiter(String delimiter) {
		this.prefs.put("delimiter", delimiter);
	}
	
	/**
	 * Checks .
	 *
	 * @return true, if consecutive delimiters should be treated as one
	 */
	public boolean isTreatConsecutiveAsOne() {
		return this.prefs.getBoolean("treatConsecutiveAsOne",true);
	}
	
	/**
	 * Sets whether consecutive delimiters should be treated as one.
	 *
	 * @param treatConsecutiveAsOne flag for treating consecutive delimiters as one
	 */
	public void setTreatConsecutiveAsOne(boolean treatConsecutiveAsOne) {
		this.prefs.putBoolean("treatConsecutiveAsOne",treatConsecutiveAsOne);
	}
	
	
	/** Gets the character that is used as delimiter when importing data. Only used to 
	 * store this information internally. 
	 * 
	 * @return the user-defined delimiter
	 */
	public String getOtherDelimiter() {
		return this.prefs.get("otherDelimiter","");
	}
	
	/**
	 * Sets the user-defined delimiter.
	 *
	 * @param otherDelimiter the user-defined delimiter
	 */
	public void setOtherDelimiter(String otherDelimiter) {
		this.prefs.put("otherDelimiter", otherDelimiter);
	}
	
	/**
	 * Checks if filtered designs should be shown.
	 *
	 * @return true, if filtered designs are shown
	 */
	public boolean isShowFilteredDesigns() {
		return this.prefs.getBoolean("showFilteredDesigns",false);
	}
	
	/**
	 * Sets whether filtered designs should be shown.
	 *
	 * @param showFilteredDesigns flag to set whether filtered designs should be shown.
	 */
	public void setShowFilteredDesigns(boolean showFilteredDesigns) {
		this.prefs.putBoolean("showFilteredDesigns",showFilteredDesigns);
	}
	
	/**
	 * Gets the active design default color.
	 *
	 * @return the active design default color
	 */
	public Color getActiveDesignDefaultColor() {
		int r=this.prefs.getInt("activeDesignDefaultColorRed", 0);
		int g=this.prefs.getInt("activeDesignDefaultColorGreen", 255);
		int b=this.prefs.getInt("activeDesignDefaultColorBlue", 0);
		return new Color(r,g,b);
	}
	
	/**
	 * Sets the active design default color.
	 *
	 * @param activeDesignDefaultColor the new active design default color
	 */
	public void setActiveDesignDefaultColor(Color activeDesignDefaultColor) {
		this.prefs.putInt("activeDesignDefaultColorRed", activeDesignDefaultColor.getRed());
		this.prefs.putInt("activeDesignDefaultColorGreen", activeDesignDefaultColor.getGreen());
		this.prefs.putInt("activeDesignDefaultColorBlue", activeDesignDefaultColor.getBlue());
	}
	
	/**
	 * Gets the filtered design default color.
	 *
	 * @return the filtered design default color
	 */
	public Color getFilteredDesignDefaultColor() {
		int r=this.prefs.getInt("inActiveDesignDefaultColorRed", 100);
		int g=this.prefs.getInt("inActiveDesignDefaultColorGreen", 100);
		int b=this.prefs.getInt("inActiveDesignDefaultColorBlue", 100);
		return new Color(r,g,b);
	}
	
	/**
	 * Sets the in active design default color.
	 *
	 * @param inActiveDesignDefaultColor the new in active design default color
	 */
	public void setInActiveDesignDefaultColor(Color inActiveDesignDefaultColor) {
		this.prefs.putInt("inActiveDesignDefaultColorRed", inActiveDesignDefaultColor.getRed());
		this.prefs.putInt("inActiveDesignDefaultColorGreen", inActiveDesignDefaultColor.getGreen());
		this.prefs.putInt("inActiveDesignDefaultColorBlue", inActiveDesignDefaultColor.getBlue());
	}
	
	/**
	 * Checks if design id numbers are shown left to the left-most axis.
	 *
	 * @return whether design id numbers are shown left to the left-most axis.
	 */
	public boolean isShowDesignIDs() {
		return this.prefs.getBoolean("showDesignIDs",true);
	}
	
	/**
	 * Specifies whether design id numbers are shown left to the left-most axis.
	 *
	 * @param showDesignIDs specifies whether design id numbers are shown left to the left-most axis.
	 */
	public void setShowDesignIDs(boolean showDesignIDs) {
		this.prefs.putBoolean("showDesignIDs",showDesignIDs);
	}
	
	/**
	 * Gets the design id font size.
	 *
	 * @return the design id font size
	 */
	public int getDesignIDFontSize() {
		return this.prefs.getInt("designIDFontSize",10);
	}
	
	/**
	 * Sets the design id font size.
	 *
	 * @param designIDFontSize the new design id font size
	 */
	public void setDesignIDFontSize(int designIDFontSize) {
		this.prefs.putInt("designIDFontSize", designIDFontSize);
	}

	/**
	 * Gets the default background color.
	 *
	 * @return the default background color
	 */
	public Color getDefaultBackgroundColor() {
		int r=this.prefs.getInt("backgroundColorRed", 255);
		int g=this.prefs.getInt("backgroundColorGreen", 255);
		int b=this.prefs.getInt("backgroundColorBlue", 255);
		return new Color(r,g,b);
	}
	
	/**
	 * Sets the default background color.
	 *
	 * @param backgroundColor the new default background color
	 */
	public void setDefaultBackgroundColor(Color backgroundColor) {
		this.prefs.putInt("backgroundColorRed", backgroundColor.getRed());
		this.prefs.putInt("backgroundColorGreen", backgroundColor.getGreen());
		this.prefs.putInt("backgroundColorBlue", backgroundColor.getBlue());
	}
	
	/** True, when filters are inverted. This means that designs between the filters are shown, while designs
	 * above the top filter and below the bottom filter are not. 
	 *
	 * @return true, if filters are inverted.
	 */
	public boolean isFilterInverted() {
		return this.prefs.getBoolean("filterInverted",false);
	}
	
	/**
	 * Specifies whether filters inverted.
	 *
	 * @param filterInverted Specifies whether filters inverted.
	 */
	public void setFilterInverted(boolean filterInverted) {
		this.prefs.putBoolean("filterInverted",filterInverted);
	}

	
	/**
	 * Checks whether axes are inverted.
	 *
	 * @return true, if axes are inverted
	 */
	public boolean isAxisInverted() {
		return this.prefs.getBoolean("axisInverted",false);
	}
	
	/**
	 * Specifies whether axes are inverted.
	 *
	 * @param axisInverted Specifies whether axes are inverted.
	 */
	public void setAxisInverted(boolean axisInverted) {
		this.prefs.putBoolean("axisInverted",axisInverted);
	}
	
	/**
	 * Checks if axes are autofitted.
	 *
	 * @return true, if axes are autofitted.
	 */
	public boolean isAutoFitAxis() {
		return this.prefs.getBoolean("autoFitAxis",true);
	}
	
	/**
	 * Specifies whether axes should be autofitted.
	 *
	 * @param autoFitAxis Specifies whether axes should be autofitted.
	 */
	public void setAutoFitAxis(boolean autoFitAxis) {
		this.prefs.putBoolean("autoFitAxis",autoFitAxis);
	}

	
	/**
	 * Gets the axis default minimum value.
	 *
	 * @return the axis default minimum value
	 */
	public double getAxisDefaultMin() {
		return this.prefs.getDouble("axisDefaultMin",-10);
	}
	
	/**
	 * Sets the axis default minimum value.
	 *
	 * @param axisDefaultMin the new axis default minimum value
	 */
	public void setAxisDefaultMin(double axisDefaultMin) 
	{
		this.prefs.putDouble("axisDefaultMin",axisDefaultMin);
	}

	/**
	 * Gets the axis default maximum value.
	 *
	 * @return the axis default maximum value
	 */
	public double getAxisDefaultMax() {
		return this.prefs.getDouble("axisDefaultMax", 10.0);
	}
	
	/**
	 * Sets the axis default max.
	 *
	 * @param axisDefaultMax the new axis default max
	 */
	public void setAxisDefaultMax(double axisDefaultMax) {
		this.prefs.putDouble("axisDefaultMax", axisDefaultMax);
	}
	
	/** Returns the number format locale to be used for parsing
	 *  data.
	 * 
	 * @return the locale
	 */
	public Locale getLocale() {
		int locale = this.prefs.getInt("locale",LOCALE_US);
		switch (locale)
		{
		case(LOCALE_DE):{return Locale.GERMANY;}
		default:{return Locale.US;}
		}
		
		
	}
	
	/**
	 * Sets the number format string to be used for both parsing
	 *  data.
	 *  
	 * @param locale the Number Format locale
	 */
	public void setLocale(int locale) {
		this.prefs.putInt("locale", locale);
	}	
	
	/**
	 * Restores all default settings.
	 */
	public void resetToDefault()
	{
		try {
			this.prefs.clear();
		} catch (BackingStoreException e) {
			// TODO Auto-generated catch block
			System.err.println(e.getMessage());
		}
		this.setLicenseAccepted(true);
	}
	
	/**
	 * Prints debug information to stdout when printLog is set to true.
	 *
	 * @param message the message
	 */
	private void log(String message)
	{
		if(UserPreferences.printLog && Main.isLoggingEnabled())
		{
			System.out.println(this.getClass().getName()+"."+message);
		}
	}
}
