/*
 *  Copyright 2012, Enguerrand de Rochefort
 * 
 * This file is part of xdat.
 *
 * xdat is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * xdat is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with xdat.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */


package gui.panels;

import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import javax.swing.JPanel;

import xdat.Main;

import chart.Axis;
import chart.Chart;
import chart.Filter;
import data.DataSheet;
import data.Design;


/**
 * Panel that is used to display a {@link chart.Chart}.
 */
public class ChartPanel 
extends JPanel 
implements MouseMotionListener, MouseListener
{	
	/** The version tracking unique identifier for Serialization. */
	static final long serialVersionUID = 0003;
	
	/** Flag to enable debug message printing for this class. */
	static final boolean printLog=false;
	
	/** The chart. */
	private Chart chart;
	
	/** The data sheet. */
	private DataSheet dataSheet;
	
	/** The top margin. */
	private int marginTop = 20;
	
	/** The bottom margin. */
	private int marginBottom = 80;
	
	/** The left margin. */
	private int marginLeft = 80;
	
	/** The right margin. */
	private int marginRight = 20;
	
	/** Reference to a filter that is currently being dragged by the user. */
	private Filter draggedFilter;
	
	/** When the user is dragging a filter, the initial x position is stored in this field. */
	private int dragStartX;
	
	/** When the user is dragging a filter, the initial y position is stored in this field. */
	private int dragStartY;
	
	/** Stores how far up or down the mouse was dragged for further use. */
	private int dragOffsetY;
	
	/**
	 * Instantiates a new chart panel.
	 *
	 * @param dataSheet the data sheet
	 * @param chart the chart
	 */
	public ChartPanel(DataSheet dataSheet, Chart chart) {
		log("constructor called");
		this.addMouseListener(this);
		this.addMouseMotionListener(this);
		this.dataSheet = dataSheet;
		this.chart = chart;
	}

	/**
	 * Overridden to implement the painting of the chart.
	 * 
	 * @see javax.swing.JComponent#paintComponent(java.awt.Graphics)
	 */
	public void paintComponent(Graphics g)
	{
		super.paintComponent(g);
		log("paintComponent: nr of designs is "+chart.getDataSheet().getDesignCount());
		this.setBackground(chart.getBackGroundColor());
		this.drawPlotFieldBackground(g);
		this.drawDesigns(g);
		this.drawAxes(g);

	}	
		
	/**
	 * Draws the plot field background.
	 *
	 * @param g the graphics object
	 */
	public void drawPlotFieldBackground(Graphics g)
	{
		g.setColor(this.chart.getBackGroundColor());
		g.fillRect(0, 0, this.getWidth(), this.getHeight());
	}	
	
	/**
	 * Draws the lines representing the designs.
	 *
	 * @param g the graphics object
	 */
	public void drawDesigns(Graphics g)
	{
		int axisTopPos = this.chart.getAxisTopPos();
		int designLabelFontSize = this.chart.getDesignLabelFontSize();
		int axisCount = this.chart.getAxisCount();
		double[] axisRanges = new double[axisCount];
		int[] axisHeights = new int[axisCount];
		int[] axisWidths = new int[axisCount];
		double[] axisMaxValues = new double[axisCount];
		double [] axisMinValues = new double[axisCount];
		int[] axisTicLabelFontsizes = new int[axisCount];
		boolean[] axisActiveFlags = new boolean[axisCount];
		boolean[] axisInversionFlags = new boolean[axisCount];
		for(int i=0; i<axisCount; i++) // read all the display settings and put them into arrays to improve rendering speed of the chart
		{
			axisRanges[i] = chart.getAxis(i).getMax()-chart.getAxis(i).getMin();
			axisHeights[i] = chart.getAxis(i).getHeight();
			axisWidths[i]  = chart.getAxis(i).getWidth();
			axisMaxValues[i] = chart.getAxis(i).getMax();
			axisMinValues[i] = chart.getAxis(i).getMin();
			axisTicLabelFontsizes[i]  = chart.getAxis(i).getTicLabelFontSize();
			axisActiveFlags[i] = chart.getAxis(i).isActive();
			axisInversionFlags[i] = chart.getAxis(i).isAxisInverted();
		}
		
		for(int designID=0; designID<this.dataSheet.getDesignCount(); designID++)		// draw all designs
		{
			Design currentDesign = this.dataSheet.getDesign(designID);
			log("drawDesigns: currentDesign.isInsideBounds(chart) = "+currentDesign.isInsideBounds(chart));
			if(!currentDesign.isInsideBounds(chart))	// do not draw design if it is not inside bounds of the chart
				continue;
			boolean firstAxisDrawn = false;
			boolean currentDesignClusterActive = true;
			if (currentDesign.getCluster() != null)		// determine if design belongs to an active cluster
			{
				currentDesignClusterActive = currentDesign.getCluster().isActive();
			}

			log("drawDesigns: currentDesignClusterActive = "+currentDesignClusterActive);
			boolean currentDesignActive = true;
			currentDesignActive = currentDesign.isActive(this.chart);	// determine if current design is active
			log("drawDesigns: currentDesign.isActive(this.chart) = "+currentDesign.isActive(this.chart));
			if((currentDesignActive || this.chart.isShowFilteredDesigns()) && (currentDesignClusterActive))  // only draw design if the cluster is active and the design is active (or inactive design drawing is active)
			{
				g.setColor(this.chart.getDesignColor(currentDesign, currentDesignActive));
				int xPositionCurrent = marginLeft;
				int yPositionCurrent = axisTopPos;	
				int xPositionLast = xPositionCurrent;
				int yPositionLast;			
				for(int i=0; i<axisCount; i++)
				{
					int yPosition = axisTopPos;
					if(axisActiveFlags[i])
					{
						double value = currentDesign.getDoubleValue(dataSheet.getParameter(i));
						
						int yPositionRelToBottom;
						if(axisRanges[i]==0)
						{
							yPositionRelToBottom = (int)(axisHeights[i]*0.5);
						}
						else
						{
							double ratio ;
							if(axisInversionFlags[i])
							{
								ratio = (axisMaxValues[i] - value ) / axisRanges[i];
							}
							else
							{
								ratio = (value -axisMinValues[i] ) / axisRanges[i];
							}
							yPositionRelToBottom = (int)(axisHeights[i]*ratio);
						}

							
						yPositionLast = yPositionCurrent;	
						yPositionCurrent = yPosition+(axisHeights[i])-yPositionRelToBottom;
						
						if (firstAxisDrawn)
						{
							xPositionCurrent = xPositionCurrent + (int)(axisWidths[i]*0.5);
							g.drawLine(xPositionLast, yPositionLast, xPositionCurrent, yPositionCurrent);
						}
						else
						{
							firstAxisDrawn=true;
							if(this.chart.isShowDesignIDs())
							{
								FontMetrics fm = g.getFontMetrics();
								g.setFont(new Font("SansSerif",Font.PLAIN,designLabelFontSize));					
								g.drawString(
										Integer.toString(currentDesign.getId()),
										xPositionCurrent - 5 - fm.stringWidth(Integer.toString(currentDesign.getId())),
										yPositionCurrent + (int)(0.5*chart.getAxis(i).getTicLabelFontSize())
								); 
							}
						}
						xPositionLast = xPositionCurrent;
						xPositionCurrent = xPositionCurrent + (int)(axisWidths[i]*0.5);		
					}
				}
			}
		}
	}
	
	/**
	 * Draws the axes.
	 *
	 * @param g the graphics object
	 */
	public void drawAxes(Graphics g)
	{
		int xPosition = marginLeft;
		int yPosition = this.chart.getAxisTopPos();
		FontMetrics fm = g.getFontMetrics();
		Axis lastAxis=null;
		Axis currentAxis;
		for(int i=0; i<this.chart.getAxisCount(); i++)
		{
			log("drawing axis "+chart.getAxis(i).getName());
			if(chart.getAxis(i).isActive())
			{
				//axes
				currentAxis = chart.getAxis(i);
				if (null!=lastAxis)
				{
					xPosition = xPosition + (int)(lastAxis.getWidth()*0.5) + (int)(currentAxis.getWidth()*0.5);
				}		
				
				String axisLabel = currentAxis.getName();
				int slenX = fm.stringWidth(axisLabel);
				g.setFont(new Font("SansSerif",Font.PLAIN,currentAxis.getAxisLabelFontSize()));

				
				g.setColor(currentAxis.getAxisLabelFontColor());
				g.drawString(
						axisLabel,
						xPosition - (int)(0.5*slenX),
						this.chart.getMaxAxisLabelFontSize()+this.chart.getTopMargin()
				); 

				g.setColor(currentAxis.getAxisColor());
				g.drawLine(xPosition, yPosition, xPosition, yPosition+(currentAxis.getHeight()));
				
				//Filters
				
				Filter uf = currentAxis.getUpperFilter();
				Filter lf = currentAxis.getLowerFilter();

				uf.setXPos(xPosition);
				lf.setXPos(xPosition);
				
				g.setColor(chart.getFilterColor());
				g.drawLine(uf.getXPos(), uf.getYPos(), uf.getXPos()-chart.getFilterWidth(), uf.getYPos()-chart.getFilterHeight());
				g.drawLine(uf.getXPos(), uf.getYPos(), uf.getXPos()+chart.getFilterWidth(), uf.getYPos()-chart.getFilterHeight());	
				g.drawLine(uf.getXPos()-chart.getFilterWidth(), uf.getYPos()-chart.getFilterHeight(), uf.getXPos()+chart.getFilterWidth(), uf.getYPos()-chart.getFilterHeight());	
				
				g.drawLine(lf.getXPos(), lf.getYPos(), lf.getXPos()-chart.getFilterWidth(), lf.getYPos()+chart.getFilterHeight());
				g.drawLine(lf.getXPos(), lf.getYPos(), lf.getXPos()+chart.getFilterWidth(), lf.getYPos()+chart.getFilterHeight());	
				g.drawLine(lf.getXPos()-chart.getFilterWidth(), lf.getYPos()+chart.getFilterHeight(), lf.getXPos()+chart.getFilterWidth(), lf.getYPos()+chart.getFilterHeight());	

				g.setFont(new Font("SansSerif",Font.PLAIN,currentAxis.getTicLabelFontSize()));
				log("Font size: "+currentAxis.getTicLabelFontSize());
				if((uf==this.draggedFilter || lf==this.draggedFilter) && currentAxis.getParameter().isNumeric())
				{
					g.drawString(
								String.format(currentAxis.getTicLabelFormat(),this.draggedFilter.getValue()), 
								this.draggedFilter.getXPos()+chart.getFilterWidth() +4, 
								this.draggedFilter.getYPos()-chart.getFilterHeight()
								);					
				}
				
				if(null!=lastAxis)
				{
					g.drawLine(lastAxis.getUpperFilter().getXPos(), lastAxis.getUpperFilter().getYPos(), uf.getXPos(), uf.getYPos());
					g.drawLine(lastAxis.getLowerFilter().getXPos(), lastAxis.getLowerFilter().getYPos(), lf.getXPos(), lf.getYPos());									
				}
				
				//tics
				
				int ticSize = currentAxis.getTicLength();
				int ticCount = currentAxis.getTicCount();
				double ticSpacing; // must be double to avoid large round off errors
				if(ticCount>1)
					ticSpacing = currentAxis.getHeight()/ ((double)(ticCount-1));
				else
					ticSpacing = 0;
				double axisRange = currentAxis.getRange();
				double ticValueDifference = axisRange / ((double)(ticCount-1));
				for(int ticID = 0; ticID < ticCount; ticID++)
				{
					int currentTicYPos;
					if(currentAxis.isAxisInverted())
						currentTicYPos = yPosition+currentAxis.getHeight()-(int)(ticID*ticSpacing);
					else
						currentTicYPos = yPosition+(int)(ticID*ticSpacing);
					g.setColor(currentAxis.getAxisColor());
					if(ticCount>1)
						g.drawLine(xPosition, currentTicYPos, xPosition+ticSize, currentTicYPos);
					else
						g.drawLine(xPosition, yPosition+(int)(currentAxis.getHeight()/ 2), xPosition+ticSize, yPosition+(int)(currentAxis.getHeight()/ 2));
					
					
					g.setColor(currentAxis.getAxisTicLabelFontColor());
					
					String ticLabel;
					g.setFont(new Font("SansSerif",Font.PLAIN,currentAxis.getTicLabelFontSize()));
					if(currentAxis.getParameter().isNumeric())
					{
						Double ticValue;
						if(ticCount>1)
						{
							ticValue = currentAxis.getMax()-ticValueDifference*ticID;
							ticLabel = String.format(currentAxis.getTicLabelFormat(),ticValue);	
							g.drawString(
									ticLabel,
									xPosition + ticSize + 7 ,
									currentTicYPos + (int)(0.5*currentAxis.getTicLabelFontSize())							
									);
						}
						else
						{
							ticValue = currentAxis.getMax();
							ticLabel = String.format(currentAxis.getTicLabelFormat(),ticValue);	
							g.drawString(
									ticLabel,
									xPosition + 2*ticSize ,
									yPosition+((int)(currentAxis.getHeight()/ 2)) + (int)(0.5*currentAxis.getTicLabelFontSize())							
									);
						}
											
					}
					else
					{
						if(ticCount>1)
						{		
							ticLabel = currentAxis.getParameter().getStringValueOf(currentAxis.getMax()-ticValueDifference*ticID);			
							g.drawString(
									ticLabel,
									xPosition + 2*ticSize ,
									currentTicYPos + (int)(0.5*currentAxis.getTicLabelFontSize())							
									);
						}
						else
						{
							ticLabel = currentAxis.getParameter().getStringValueOf(currentAxis.getMax());				
							g.drawString(
									ticLabel,
									xPosition + 2*ticSize ,
									yPosition+((int)(currentAxis.getHeight()/ 2)) + (int)(0.5*currentAxis.getTicLabelFontSize())							
									);
						}					
					}
				}

				lastAxis=currentAxis;
			}
		}		
	}

	/* (non-Javadoc)
	 * @see java.awt.event.MouseListener#mouseClicked(java.awt.event.MouseEvent)
	 */
	public void mouseClicked(MouseEvent e)
	{	 
	}
	
	/* (non-Javadoc)
	 * @see java.awt.event.MouseListener#mouseEntered(java.awt.event.MouseEvent)
	 */
	public void mouseEntered(MouseEvent e)
	{		 
	}
	
	/* (non-Javadoc)
	 * @see java.awt.event.MouseListener#mouseExited(java.awt.event.MouseEvent)
	 */
	public void mouseExited(MouseEvent e)
	{	 
	}

	/* (non-Javadoc)
	 * @see java.awt.event.MouseListener#mousePressed(java.awt.event.MouseEvent)
	 */
	public void mousePressed(MouseEvent e)
	{
		dragStartX = e.getX();
		dragStartY = e.getY();
		for(int i=0; i<this.chart.getAxisCount(); i++)
		{
			Filter uf = this.chart.getAxis(i).getUpperFilter();
			Filter lf = this.chart.getAxis(i).getLowerFilter();
			if	// check whether the drag operation started on the upper filter
			(
				dragStartY >= uf.getYPos()-chart.getFilterHeight() &&
				dragStartY <= uf.getYPos() &&
				dragStartX >= uf.getXPos() - chart.getFilterWidth() &&
				dragStartX <= uf.getXPos() + chart.getFilterWidth()
			)
			{
				this.draggedFilter = uf;
				this.dragOffsetY = uf.getYPos() - dragStartY;
			}
			else if	// check whether the drag operation started on the lower filter
			(
					dragStartY >= lf.getYPos() &&
					dragStartY <= lf.getYPos() + chart.getFilterHeight() &&
					dragStartX >= lf.getXPos() - chart.getFilterWidth() &&
					dragStartX <= lf.getXPos() + chart.getFilterWidth()
				)
				{
					this.draggedFilter = lf;
					this.dragOffsetY = lf.getYPos()-dragStartY;
				}
		}
	}
	
	/* (non-Javadoc)
	 * @see java.awt.event.MouseListener#mouseReleased(java.awt.event.MouseEvent)
	 */
	public void mouseReleased(MouseEvent e)
	{
		this.draggedFilter = null;
		repaint();
	}

	/* (non-Javadoc)
	 * @see java.awt.event.MouseMotionListener#mouseMoved(java.awt.event.MouseEvent)
	 */
	public void mouseMoved(MouseEvent e)
	{
	}
	
	/* (non-Javadoc)
	 * @see java.awt.event.MouseMotionListener#mouseDragged(java.awt.event.MouseEvent)
	 */
	public void mouseDragged(MouseEvent e)
	{
		if(this.draggedFilter!=null)
		{
			// try to make the filter follow the drag operation, but always keep it within axis boundaries and opposite filter
			this.draggedFilter.setYPos(Math.max(Math.min(e.getY()+this.dragOffsetY, this.draggedFilter.getLowestPos()),this.draggedFilter.getHighestPos()));
			repaint();
		}
			

	}
	
	/* (non-Javadoc)
	 * @see javax.swing.JComponent#getPreferredSize()
	 */
	public Dimension getPreferredSize()
	{
		int width = marginLeft+marginRight+chart.getWidth();
		int height = marginTop+marginBottom+chart.getHeight();
		Dimension preferredSize = new Dimension(width,height);

		return preferredSize;
	}
	
	/**
	 * Prints debug information to stdout when printLog is set to true.
	 *
	 * @param message the message
	 */
	private void log(String message)
	{
		if(ChartPanel.printLog && Main.isLoggingEnabled())
		{
			System.out.println(this.getClass().getName()+"."+message);
		}
	}

}
