/*
 *  Copyright 2012, Enguerrand de Rochefort
 * 
 * This file is part of xdat.
 *
 * xdat is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * xdat is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with xdat.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package gui.panels;

import gui.buttons.ColorChoiceButton;
import gui.buttons.MinMaxSpinnerModel;
import gui.dialogs.DisplaySettingsDialog;
import gui.frames.ChartFrame;

import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.GridLayout;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JSpinner;

import xdat.Main;

import chart.Chart;

import actionListeners.ChartDisplaySettingsActionListener;
import actionListeners.ChartSpecificDisplaySettingsDialogActionListener;
import actionListeners.DefaultDisplaySettingsDialogActionListener;


/**
 * Panel to modify display settings for a {@link chart.Chart}.
 */
public class ChartDisplaySettingsPanel 
extends JPanel 
{
	/** The version tracking unique identifier for Serialization. */
	static final long serialVersionUID = 0000;
	
	/** Flag to enable debug message printing for this class. */
	static final boolean printLog=false;
	
	/** The main window. */
	private Main mainWindow;	
	
	/** The dialog on which the panel is located. */
	private DisplaySettingsDialog dialog;
	
	/** The chart frame to which the settings apply. */
	private ChartFrame chartFrame;
	
	/** The action listener */
	private ChartDisplaySettingsActionListener cmd;
	
	/** The axis height spinner. */
	private JSpinner axisHeightSpinner = new JSpinner(new MinMaxSpinnerModel(0,3000));
	
	/** The back ground color button. */
	private ColorChoiceButton backGroundColorButton;
	
	/** The active design color button. */
	private ColorChoiceButton activeDesignColorButton;
	
	/** The filtered design color button. */
	private ColorChoiceButton filteredDesignColorButton;
	
	/** The filter color button. */
	private ColorChoiceButton filterColorButton;
	
	/** The showfiltered designs true button. */
	private JRadioButton showfilteredDesignsTrueButton = new JRadioButton("Yes");
	
	/** The showfiltered designs false button. */
	private JRadioButton showfilteredDesignsFalseButton = new JRadioButton("No");
	
	/** The show design IDs true button. */
	private JRadioButton showDesignIDsTrueButton = new JRadioButton("Yes");
	
	/** The show design IDs false button. */
	private JRadioButton showDesignIDsFalseButton = new JRadioButton("No");
	
	/** The show filtered designs button group. */
	private ButtonGroup showfilteredDesignsButtonGroup = new ButtonGroup();
	
	/** The show design IDs button group. */
	private ButtonGroup showDesignIDsButtonGroup = new ButtonGroup();
	
	/** The design label font size spinner. */
	private JSpinner designLabelFontSizeSpinner = new JSpinner(new MinMaxSpinnerModel(0,100)); 
	
	/** The filter width spinner. */
	private JSpinner filterWidthSpinner = new JSpinner(new MinMaxSpinnerModel(1,30));
	
	/** The filter height spinner. */
	private JSpinner filterHeightSpinner = new JSpinner(new MinMaxSpinnerModel(1,60));
	
	/** The cancel button. */
	private JButton cancelButton = new JButton("Cancel");
	
	/** The ok button. */
	private JButton okButton = new JButton("Ok"); 
	
	/**
	 * Instantiates a new chart display settings panel the allows editing the default settings 
	 * in the user preferences.
	 *
	 * @param mainWindow the main window
	 * @param dialog the dialog on which the panel is located
	 * 
	 * @see DefaultDisplaySettingsDialogActionListener
	 */
	public ChartDisplaySettingsPanel(Main mainWindow, DisplaySettingsDialog dialog) 
	{
		this.mainWindow = mainWindow;
		this.dialog = dialog;
		
		buildPanel();
		// set states
		this.axisHeightSpinner.setValue(Main.getUserPreferences().getAxisHeight());
		this.setShowFilteredDesignsSelection(Main.getUserPreferences().isShowFilteredDesigns());
		this.setShowDesignIDsSelection(Main.getUserPreferences().isShowDesignIDs());
		this.designLabelFontSizeSpinner.setValue(Main.getUserPreferences().getDesignLabelFontSize());
		this.backGroundColorButton.setCurrentColor(Main.getUserPreferences().getDefaultBackgroundColor());			
		this.activeDesignColorButton.setCurrentColor(Main.getUserPreferences().getActiveDesignDefaultColor());			
		this.filteredDesignColorButton.setCurrentColor(Main.getUserPreferences().getFilteredDesignDefaultColor());			
		this.filterColorButton.setCurrentColor(Main.getUserPreferences().getFilterDefaultColor());		
		this.filterWidthSpinner.setValue(Main.getUserPreferences().getFilterWidth());
		this.filterHeightSpinner.setValue(Main.getUserPreferences().getFilterHeight());	
	
	}

	/**
	 * Instantiates a new chart display settings panel that allows editing a particular chart.
	 *
	 * @param mainWindow the main window
	 * @param dialog the dialog on which the panel is located
	 * @param chartFrame the chart which should be modified
	 * 
	 * @see ChartSpecificDisplaySettingsDialogActionListener
	 */
	public ChartDisplaySettingsPanel(Main mainWindow, DisplaySettingsDialog dialog, ChartFrame chartFrame) 
	{
		this.mainWindow = mainWindow;
		this.dialog = dialog;
		this.chartFrame = chartFrame;
		
		buildPanel();
		// set states
		this.axisHeightSpinner.setValue(chartFrame.getChart().getAxisMaxHeight());
		this.setShowFilteredDesignsSelection(chartFrame.getChart().isShowFilteredDesigns());
		this.setShowDesignIDsSelection(chartFrame.getChart().isShowDesignIDs());
		this.designLabelFontSizeSpinner.setValue(chartFrame.getChart().getDesignLabelFontSize());
		this.backGroundColorButton.setCurrentColor(chartFrame.getChart().getBackGroundColor());		
		this.activeDesignColorButton.setCurrentColor(chartFrame.getChart().getDefaultDesignColor(true));		
		this.filteredDesignColorButton.setCurrentColor(chartFrame.getChart().getDefaultDesignColor(false));			
		this.filterColorButton.setCurrentColor(chartFrame.getChart().getFilterColor());	 	
		this.filterWidthSpinner.setValue(chartFrame.getChart().getFilterWidth());
		this.filterHeightSpinner.setValue(chartFrame.getChart().getFilterHeight());				
	
	}
	
	/**
	 * Builds the panel.
	 */
	private void buildPanel() {
		// create components
		
		this.backGroundColorButton = new ColorChoiceButton(Main.getUserPreferences().getDefaultBackgroundColor(), "Background Color");		
		this.activeDesignColorButton = new ColorChoiceButton(Main.getUserPreferences().getActiveDesignDefaultColor(), "Active Design Color");		
		this.filteredDesignColorButton = new ColorChoiceButton(Main.getUserPreferences().getFilteredDesignDefaultColor(), "Filtered Design Color");		
		this.filterColorButton = new ColorChoiceButton(Main.getUserPreferences().getFilterDefaultColor(), "Filter Color");			
		TitledSubPanel contentPanel = new TitledSubPanel("");
		JPanel contentInnerPanel = new JPanel(new BorderLayout());
		JPanel labelPanel = new JPanel(new GridLayout(10,1));
		JPanel controlsPanel = new JPanel(new GridLayout(10,1));
		TitledSubPanel buttonsPanel = new TitledSubPanel("");
		JPanel showfilteredDesignsRadioButtonsPanel = new JPanel(new GridLayout(1,2));
		JPanel showDesignIDsRadioButtonsPanel = new JPanel(new GridLayout(1,2));

		JLabel axisHeightLabel = new JLabel("Axis Height");
		JLabel backGroundColorLabel = new JLabel("Background Color");
		JLabel activeDesignColorLabel = new JLabel("Active Design Color");
		JLabel filteredDesignColorLabel = new JLabel("Filtered Design Color");
		JLabel filterColorLabel = new JLabel("Filter Color");
		JLabel showfilteredDesignsLabel = new JLabel("Show filtered Designs");
		JLabel showDesignIDsLabel = new JLabel("Show Design IDs");
		showfilteredDesignsButtonGroup.add(showfilteredDesignsTrueButton);
		showfilteredDesignsButtonGroup.add(showfilteredDesignsFalseButton);
		showDesignIDsButtonGroup.add(showDesignIDsTrueButton);
		showDesignIDsButtonGroup.add(showDesignIDsFalseButton);
		JLabel designLabelFontSizeLabel = new JLabel("Design Label Font Size ");
		JLabel filterWidthLabel = new JLabel("Filter Symbol Width");
		JLabel filterHeightLabel = new JLabel("Filter Symbol Height");

		JPanel cancelButtonPanel = new JPanel();
		JPanel okButtonPanel = new JPanel();

		// set Layouts
		this.setLayout(new BorderLayout());
		contentPanel.setLayout(new BorderLayout());		
		cancelButtonPanel.setLayout(new FlowLayout(FlowLayout.RIGHT));
		okButtonPanel.setLayout(new FlowLayout(FlowLayout.LEFT));
		buttonsPanel.setLayout(new GridLayout(1,2));
		
		// add components
		this.add(contentPanel, BorderLayout.CENTER);
		contentPanel.add(contentInnerPanel, BorderLayout.NORTH);
		contentInnerPanel.add(labelPanel, BorderLayout.CENTER);
		contentInnerPanel.add(controlsPanel, BorderLayout.EAST);
		this.add(buttonsPanel,BorderLayout.SOUTH);
		labelPanel.add(axisHeightLabel);
		controlsPanel.add(axisHeightSpinner);
		labelPanel.add(backGroundColorLabel);
		JPanel backGroundColorButtonPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
		backGroundColorButtonPanel.add(backGroundColorButton);
		controlsPanel.add(backGroundColorButtonPanel);
		labelPanel.add(activeDesignColorLabel);
		JPanel activeDesignColorButtonPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
		controlsPanel.add(activeDesignColorButtonPanel);
		activeDesignColorButtonPanel.add(activeDesignColorButton);	
		labelPanel.add(filteredDesignColorLabel);
		JPanel filteredDesignColorButtonPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
		filteredDesignColorButtonPanel.add(filteredDesignColorButton);
		controlsPanel.add(filteredDesignColorButtonPanel);
		labelPanel.add(filterColorLabel);
		JPanel filterColorButtonPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
		filterColorButtonPanel.add(filterColorButton);
		controlsPanel.add(filterColorButtonPanel);
		labelPanel.add(showfilteredDesignsLabel);
		controlsPanel.add(showfilteredDesignsRadioButtonsPanel);
		showfilteredDesignsRadioButtonsPanel.add(showfilteredDesignsTrueButton);
		showfilteredDesignsRadioButtonsPanel.add(showfilteredDesignsFalseButton);
		labelPanel.add(showDesignIDsLabel);
		controlsPanel.add(showDesignIDsRadioButtonsPanel);
		labelPanel.add(designLabelFontSizeLabel);
		controlsPanel.add(designLabelFontSizeSpinner);
		showDesignIDsRadioButtonsPanel.add(showDesignIDsTrueButton);
		showDesignIDsRadioButtonsPanel.add(showDesignIDsFalseButton);
		showfilteredDesignsTrueButton.setActionCommand("showfilteredDesignsTrue");
		showfilteredDesignsFalseButton.setActionCommand("showfilteredDesignsFalse");
		showDesignIDsTrueButton.setActionCommand("showDesignIDsTrue");
		showDesignIDsFalseButton.setActionCommand("showDesignIDsFalse");
		labelPanel.add(filterWidthLabel);
		controlsPanel.add(filterWidthSpinner);
		labelPanel.add(filterHeightLabel);
		controlsPanel.add(filterHeightSpinner);
		// buttons panel
		buttonsPanel.add(cancelButtonPanel);
		buttonsPanel.add(okButtonPanel);
		cancelButtonPanel.add(cancelButton);
		okButtonPanel.add(okButton);


		
	}
	
	/**
	 * Sets the action listener.
	 *
	 * @param cmd the new action listener
	 */
	public void setActionListener (ChartDisplaySettingsActionListener cmd)
	{
		this.cmd = cmd;
		backGroundColorButton.addActionListener(cmd);
		activeDesignColorButton.addActionListener(cmd);
		filteredDesignColorButton.addActionListener(cmd);
		filterColorButton.addActionListener(cmd);
		showfilteredDesignsTrueButton.addActionListener(cmd);
		showfilteredDesignsFalseButton.addActionListener(cmd);
		showDesignIDsTrueButton.addActionListener(cmd);
		showDesignIDsFalseButton.addActionListener(cmd);
	}
	
	/**
	 * Tells the panel that the settings should be applied to the user preferences.
	 * 
	 * @see DefaultDisplaySettingsDialogActionListener
	 */
	public void setOkCancelButtonTargetDefaultSettings ()
	{
		DefaultDisplaySettingsDialogActionListener cmd = new DefaultDisplaySettingsDialogActionListener(dialog);
		log("setOkCancelButtonTargetDefaultSettings called");
		cancelButton.addActionListener(cmd);
		okButton.addActionListener(cmd);	
	}	
	
	/**
	 * Tells the panel that the settings should be applied to the the chart specified in the argument.
	 *
	 * @param chart specifies which chart the settings should be applied to.
	 * 
	 * @see ChartSpecificDisplaySettingsDialogActionListener
	 */
	public void setOkCancelButtonTargetChart (Chart chart)
	{
		log("setOkCancelButtonTargetChart called");
		cancelButton.addActionListener(new ChartSpecificDisplaySettingsDialogActionListener(this.mainWindow, dialog, chart, chartFrame));
		okButton.addActionListener(new ChartSpecificDisplaySettingsDialogActionListener(this.mainWindow, dialog, chart, chartFrame));		
	}	
	
	/**
	 * Gets the chart display settings action listener.
	 *
	 * @return the chart display settings action listener
	 */
	public ChartDisplaySettingsActionListener getChartDisplaySettingsActionListener()
	{
		return this.cmd;
	}

	/**
	 * Gets the show filtered designs selection.
	 *
	 * @return the show filtered designs selection
	 */
	public boolean getShowFilteredDesignsSelection()
	{
		if(showfilteredDesignsTrueButton.getModel().equals(showfilteredDesignsButtonGroup.getSelection()))
		{
			return true;
		}
		else
		{
			return false;
		}
	}
	
	/**
	 * Sets the show filtered designs selection.
	 *
	 * @param showFilteredDesignsSelection the new show filtered designs selection
	 */
	public void setShowFilteredDesignsSelection(boolean showFilteredDesignsSelection)
	{
		if(showFilteredDesignsSelection)
			showfilteredDesignsButtonGroup.setSelected(showfilteredDesignsTrueButton.getModel(), true);
		else
			showfilteredDesignsButtonGroup.setSelected(showfilteredDesignsFalseButton.getModel(), true);
		
	}
	
	/**
	 * Gets the show design IDs selection.
	 *
	 * @return the show design IDs selection
	 */
	public boolean getShowDesignIDsSelection()
	{
		if(showfilteredDesignsTrueButton.getModel().equals(showfilteredDesignsButtonGroup.getSelection()))
		{
			return true;
		}
		else
		{
			return false;
		}
	}
	
	/**
	 * Sets the show design IDs selection.
	 *
	 * @param showFilteredDesignsSelection the new show design IDs selection
	 */
	public void setShowDesignIDsSelection(boolean showFilteredDesignsSelection)
	{
		log("setShowDesignIDsSelection: argument is "+showFilteredDesignsSelection);
		if(showFilteredDesignsSelection)
			showDesignIDsButtonGroup.setSelected(showDesignIDsTrueButton.getModel(), true);
		else
			showDesignIDsButtonGroup.setSelected(showDesignIDsFalseButton.getModel(), true);
		
	}
	


	/**
	 * Gets the design label font size spinner.
	 *
	 * @return the design label font size spinner
	 */
	public JSpinner getDesignLabelFontSizeSpinner() {
		return designLabelFontSizeSpinner;
	}

	/**
	 * Gets the axis height spinner.
	 *
	 * @return the axis height spinner
	 */
	public JSpinner getAxisHeightSpinner() {
		return axisHeightSpinner;
	}

	/**
	 * Gets the active design color button.
	 *
	 * @return the active design color button
	 */
	public ColorChoiceButton getActiveDesignColorButton() {
		return activeDesignColorButton;
	}

	/**
	 * Gets the back ground color button.
	 *
	 * @return the back ground color button
	 */
	public ColorChoiceButton getBackGroundColorButton() {
		return backGroundColorButton;
	}

	/**
	 * Gets the filter color button.
	 *
	 * @return the filter color button
	 */
	public ColorChoiceButton getFilterColorButton() {
		return filterColorButton;
	}

	/**
	 * Gets the filtered design color button.
	 *
	 * @return the filtered design color button
	 */
	public ColorChoiceButton getFilteredDesignColorButton() {
		return filteredDesignColorButton;
	}

	/**
	 * Gets the filter height spinner.
	 *
	 * @return the filter height spinner
	 */
	public JSpinner getFilterHeightSpinner() {
		return filterHeightSpinner;
	}

	/**
	 * Gets the filter width spinner.
	 *
	 * @return the filter width spinner
	 */
	public JSpinner getFilterWidthSpinner() {
		return filterWidthSpinner;
	}
	
	/**
	 * Prints debug information to stdout when printLog is set to true.
	 *
	 * @param message the message
	 */
	private void log(String message)
	{
		if(ChartDisplaySettingsPanel.printLog && Main.isLoggingEnabled())
		{
			System.out.println(this.getClass().getName()+"."+message);
		}
	}

}
