/*
 *  Copyright 2012, Enguerrand de Rochefort
 * 
 * This file is part of xdat.
 *
 * xdat is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * xdat is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with xdat.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package actionListeners;

import gui.panels.ChartDisplaySettingsPanel;

import java.awt.Color;
import java.awt.event.*;

import javax.swing.*;

import xdat.*;

import chart.Chart;



/**
 * ActionListener for a {@link ChartDisplaySettingsPanel} that allows to modify the Display Settings of a {@link Chart}.
 */
public class ChartDisplaySettingsActionListener 
implements ActionListener 
{
	
	/** Flag to enable debug message printing for this class. */
	static final boolean printLog=false;
	
	/** The user preferences. */
	private UserPreferences userPreferences;
	
	/** The panel on which the settings controls are located. */
	private ChartDisplaySettingsPanel panel;
	
	/** The Chart background color. 
	 * 
	 * @see Chart#getBackGroundColor()*/
	private Color backGroundColor;
	
	/** The active design color. 
	 * @see Chart#getDefaultDesignColor(boolean)*/
	private Color activeDesignColor; 
	
	/** The filtered design color. 
	 * @see Chart#getDefaultDesignColor(boolean)*/
	private Color filteredDesignColor; 
	
	/** The filter color. 
	 * @see Chart#getFilterColor()*/
	private Color filterColor; 
	
	/** The show filtered designs flag. 
	 * @see Chart#isShowFilteredDesigns()*/
	private boolean showFilteredDesigns;
	
	/** The show design IDs flag. 
	 * @see Chart#isShowDesignIDs()*/
	private boolean showDesignIDs;
	
	/**
	 * Instantiates a new chart display settings action listener to edit default settings.
	 *
	 * @param mainWindow the main window
	 * @param panel the panel
	 */
	public ChartDisplaySettingsActionListener(Main mainWindow, ChartDisplaySettingsPanel panel)
	{
		log("constructor called for default settings.");
		this.userPreferences = Main.getUserPreferences();
		this.panel = panel;
		this.backGroundColor = userPreferences.getDefaultBackgroundColor();
		this.activeDesignColor = userPreferences.getActiveDesignDefaultColor();
		this.filteredDesignColor = userPreferences.getFilteredDesignDefaultColor();
		this.filterColor = userPreferences.getFilterDefaultColor();
		this.showFilteredDesigns = userPreferences.isShowFilteredDesigns();
		this.showDesignIDs = userPreferences.isShowDesignIDs();
		
	}
	
	/**
	 * Instantiates a new chart display settings action listener to edit settings for a specific chart.
	 *
	 * @param mainWindow the main window
	 * @param panel the panel
	 * @param chart the chart
	 */
	public ChartDisplaySettingsActionListener(Main mainWindow, ChartDisplaySettingsPanel panel, Chart chart)
	{
		log("constructor called for chart specific settings.");
		this.userPreferences = Main.getUserPreferences();
		this.panel = panel;
		this.backGroundColor = chart.getBackGroundColor();
		this.activeDesignColor = chart.getDefaultDesignColor(true);
		this.filteredDesignColor = chart.getDefaultDesignColor(false);
		this.filterColor = chart.getFilterColor();
		this.showFilteredDesigns = chart.isShowFilteredDesigns();
		this.showDesignIDs = chart.isShowDesignIDs();
		
	}
	
	/* (non-Javadoc)
	 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
	 */
	public void actionPerformed(ActionEvent e)
	{
		String actionCommand = e.getActionCommand();
		if(actionCommand == "Background Color")
		{
			Color newColor = JColorChooser.showDialog(panel,"Background Color",this.backGroundColor);
			if(newColor != null)
				this.backGroundColor = newColor;
			this.panel.getBackGroundColorButton().setCurrentColor(this.backGroundColor);
			
		}
		else if(actionCommand == "Active Design Color")
		{
			Color newColor = JColorChooser.showDialog(panel,"Background Color",this.activeDesignColor);
			if(newColor != null)
				this.activeDesignColor = newColor;
			this.panel.getActiveDesignColorButton().setCurrentColor(this.activeDesignColor);
		}
		else if(actionCommand == "Filtered Design Color")
		{
			Color newColor = JColorChooser.showDialog(panel,"Background Color",this.filteredDesignColor);
			if(newColor != null)
				this.filteredDesignColor = newColor;
			this.panel.getFilteredDesignColorButton().setCurrentColor(this.filteredDesignColor);
		}
		else if(actionCommand == "Filter Color")
		{
			Color newColor = JColorChooser.showDialog(panel,"Background Color",this.filterColor);
			if(newColor != null)
				this.filterColor = newColor;
			this.panel.getFilterColorButton().setCurrentColor(this.filterColor);
		}
		else if(actionCommand == "showfilteredDesignsTrue")
		{
			this.showFilteredDesigns = true;
		}
		else if(actionCommand == "showfilteredDesignsFalse")
		{
			this.showFilteredDesigns = false;
		}
		else if(actionCommand == "showDesignIDsTrue")
		{
			this.showDesignIDs = true;
		}
		else if(actionCommand == "showDesignIDsFalse")
		{
			this.showDesignIDs = false;
		}
		else if(actionCommand == "Yes" || actionCommand == "No" )
		{
			// Do nothing
		}
		else
		{
			System.out.println("ChartDisplaySettingsActionListener: "+e.getActionCommand());
		}
	}

	/**
	 * Prints debug information to stdout when printLog is set to true.
	 *
	 * @param message the message
	 */
	private void log(String message)
	{
		if(ChartDisplaySettingsActionListener.printLog && Main.isLoggingEnabled())
		{
			System.out.println(this.getClass().getName()+"."+message);
		}
	}
	
	/**
	 * Gets the active design color.
	 *
	 * @return the active design color
	 */
	public Color getActiveDesignColor() {
		return activeDesignColor;
	}
	
	/**
	 * Gets the back ground color.
	 *
	 * @return the back ground color
	 */
	public Color getBackGroundColor() {
		return backGroundColor;
	}
	
	/**
	 * Gets the filter color.
	 *
	 * @return the filter color
	 */
	public Color getFilterColor() {
		return filterColor;
	}
	
	/**
	 * Gets the filtered design color.
	 *
	 * @return the filtered design color
	 */
	public Color getFilteredDesignColor() {
		return filteredDesignColor;
	}
	
	/**
	 * Checks if design IDs are shown.
	 *
	 * @return true, if design IDs are shown.
	 */
	public boolean isShowDesignIDs() {
		return showDesignIDs;
	}
	
	/**
	 * Checks if filtered designs are shown.
	 *
	 * @return true, if filtered designs are shown.
	 */
	public boolean isShowFilteredDesigns() {
		return showFilteredDesigns;
	}	
}
