/*
 *  Copyright 2010, Enguerrand de Rochefort
 * 
 * This file is part of xdat.
 *
 * xdat is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * xdat is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with xdat.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
package main;

import java.net.*;
import java.util.*;
import java.io.*;

/**
 * A serializable license key that stores valid MAC address and users.
 * <p>
 * This is used to ensure that every user how runs xdat for the first time
 * must accept the license agreement. (Click-wrap license)
 * 
 * @see LicenseChecker
 */
public class LicenseKey 
implements Serializable
{
	
	/** The version tracking unique identifier for Serialization. */
	static final long serialVersionUID = 10;
	
	/** Flag to enable debug message printing for this class. */
	private static final boolean printLog=false;
	
	/** The valid addresses. */
	Vector<String> validAddresses=new Vector<String>(0,1);
	
	/** The valid user names. */
	Vector<String> validUserNames=new Vector<String>(0,1);
	
	/**
	 * Instantiates a new license key.
	 */
	public LicenseKey()
	{
		log("constructor called.");
		try {
			Enumeration<NetworkInterface> ni= (Enumeration<NetworkInterface>)NetworkInterface.getNetworkInterfaces(); 

			while (ni.hasMoreElements())
			{
				byte[] address = ni.nextElement().getHardwareAddress();
				if (!macAdressToString(address).equals("00-00-00-00-00-00"))
				{
					validAddresses.add(macAdressToString(address));
					log("constructor: stored valid address "+macAdressToString(address));
				}
			}
		} catch (SocketException e) {
			System.out.println("Found exception, printing stacktrace:");
			e.printStackTrace();
		}

		this.validUserNames.add(System.getProperty("user.name"));
		log("constructor: stored valid user "+System.getProperty("user.name"));
	}

	/**
	 * Instantiates a license key that already contains addresses and usernames.
	 *
	 * @param validAddresses the valid addresses
	 * @param validUserNames the valid user names
	 */
	public LicenseKey(Vector<String> validAddresses, Vector<String> validUserNames)
	{
		this.validAddresses=validAddresses;
		this.validUserNames=validUserNames;
		this.validUserNames.add(System.getProperty("user.name"));
		log("constructor: added valid user "+System.getProperty("user.name"));
	}
	

	// found on http://www.koders.com/java/fid95863966C03EAC4031B557B2977907B1F0D1EE66.aspx
	/* Created on Jan 19, 2005
	 * 
	 * Jeopardy Firewall Management Project
	 * Copyright (C) 2005  Ozgur Akan */
	
	/**
	 * Converts a Mac address to a string.
	 *
	 * @param macAddress the mac address
	 * @return the string
	 */
	public static String macAdressToString(byte[] macAddress) {			
		char ch ='-';
		StringBuffer sb = new StringBuffer( 17 );
		for ( int i=44; i>=0; i-=4 ) {
			int nibble =  ((int)( byte2Long(macAddress) >>> i )) & 0xf;
			char nibbleChar = (char)( nibble > 9 ? nibble + ('A'-10) : nibble + '0' );
			sb.append( nibbleChar );
			if ( (i & 0x7) == 0 && i != 0 ) {
				sb.append( ch );
			}
		}
		return sb.toString();		
	}
	
	/**
	 * Converts a byte array to a long value.
	 *
	 * @param addr the byte array
	 * @return the long
	 */
	private static long byte2Long(byte addr[]) {
	    long address = 0;
		if (addr != null) {
		    if (addr.length == 6) {
			address = unsignedByteToLong(addr[5]);
			address |= (unsignedByteToLong(addr[4]) << 8);
			address |= (unsignedByteToLong(addr[3]) << 16);
			address |= (unsignedByteToLong(addr[2]) << 24);
			address |= (unsignedByteToLong(addr[1]) << 32);
			address |= (unsignedByteToLong(addr[0]) << 40);
		    } 
		} 
		return address;
	}	
	
	/**
	 * Converts an unsigned byte to a long value.
	 *
	 * @param b the byte 
	 * @return the long value
	 */
	private static long unsignedByteToLong(byte b) {
	    return (long) b & 0xFF;
	}

	/**
	 * Gets the valid addresses of this key.
	 *
	 * @return the valid addresses
	 */
	public Vector<String> getValidAddresses() {
		return validAddresses;
	}

	/**
	 * Gets the valid user names of this key.
	 *
	 * @return the valid user names
	 */
	public Vector<String> getValidUserNames() {
		return validUserNames;
	}		
	
	/**
	 * Prints debug information to stdout when printLog is set to true.
	 *
	 * @param message the message
	 */
	private void log(String message)
	{
		if(LicenseKey.printLog && Main.isLoggingEnabled())
		{
			System.out.println(this.getClass().getName()+"."+message);
		}
	}

}
