/*
 *  Copyright 2010, Enguerrand de Rochefort
 * 
 * This file is part of xdat.
 *
 * xdat is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * xdat is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with xdat.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
package main;

import gui.dialogs.*;

import java.net.*;
import java.util.*;
import java.io.*;

/**
 * Class that is able to read, check, and create license key files.
 * 
 * @see main.LicenseKey
 */
public class LicenseChecker 
implements Serializable
{
	
	/** The version tracking unique identifier for Serialization. */
	static final long serialVersionUID = 0000;
	
	/** Flag to enable debug message printing for this class. */
	private static final boolean printLog=false;
	
	/** Stores whether the license was accepted by the user. */
	private boolean licenseAccepted = false;
	
	/** The license file name. */
	private String licenseFileName ="key.lic";
	
	/**
	 * Instantiates a new license checker.
	 */
	public LicenseChecker()
	{
		log("constructor called.");
	}
	
	/**
	 * Creates a new license key file after the user confirmed that he agrees to the license.
	 *
	 * @param key the license key key
	 * @return true, if successful
	 */
	public boolean createKeyFile(LicenseKey key)
	{	
   		log("createKeyFile called. Starting dialog");
   		new LicenseDisplayDialog(this); 
   		log("createKeyFile: Dialog closed. LicenseAccepted="+licenseAccepted);	
   		if(this.licenseAccepted==true)
   		{
   			try 
   			{
   		   		FileOutputStream fs = new FileOutputStream("."+System.getProperty("file.separator")+licenseFileName);
   		   		ObjectOutputStream os = new ObjectOutputStream(fs);
   		   		os.writeObject(key);
   		   		os.close();
   		   		log("createKeyFile returning licenseAccepted = true");
   		   		return true;
   	   		} catch (IOException e) 
   	   		{
   	   			System.err.println(e.toString());
   	   		}	  			
   		}   	
   		log("createKeyFile returning licenseAccepted = "+licenseAccepted);
   		return this.licenseAccepted;	
	}

	/**
	 * Reads information from a license key file.
	 *
	 * @return the license key
	 * @throws InvalidClassException the invalid class exception
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	public LicenseKey readKeyFile()
	throws java.io.InvalidClassException, java.io.IOException
   	{
		try {
	   		FileInputStream fs = new FileInputStream("."+System.getProperty("file.separator")+licenseFileName);
	   		ObjectInputStream is = new ObjectInputStream(fs);
	   		LicenseKey readKey = (LicenseKey)is.readObject();
	   		is.close();
	   		return readKey;
		} catch (ClassNotFoundException e) {
			System.err.println(e.toString());  	
   		}

		return null;
   	}

	/**
	 * Checks whether the MAC address of this machine matches the MAC adress in the license file.
	 *
	 * @param key the license key
	 * @return true, if a match was found
	 */
	public boolean checkHostid(LicenseKey key)
	{
		boolean success = false;
		try 
		{
			Enumeration<NetworkInterface> ni= (Enumeration<NetworkInterface>)NetworkInterface.getNetworkInterfaces(); 

			while (ni.hasMoreElements())
			{
				byte[] address = ni.nextElement().getHardwareAddress();
				for(int i=0; i<key.getValidAddresses().size();i++)
				{
					log("checkHostid: Checking found MAC address "+macAdressToString(address)+" against valid address "+key.getValidAddresses().get(i));
					if(macAdressToString(address).equals(key.getValidAddresses().get(i)))
					{
						log("checkHostid: Match found, address check successful.");
						success=true;
						break;
					}
					else
					{
						log("checkHostid: Check failed.");
					}
				}
				if(success)
				{
					break;
				}
			}
			return success;
		} catch (SocketException e) 
		{
			// TODO Auto-generated catch block
			System.out.println("Found exception, printing stacktrace:");
			e.printStackTrace();

			return success;
		}		
		
	}

	/**
	 * Checks, whether the current username matches one of the usernames stored in the license key.
	 * <p>
	 * This ensures that every user on this machine has to accept the license agreement.
	 *
	 * @param key the license key
	 * @return true, if successful
	 */
	public boolean checkUsername(LicenseKey key)
	{
		boolean success = false;
		String currentUser=System.getProperty("user.name");

		for(int i=0; i<key.getValidUserNames().size();i++)
		{
			log("checkHostid: Checking user "+currentUser+" against valid user "+key.getValidUserNames().get(i));
			if(currentUser.equals(key.getValidUserNames().get(i)))
			{
				log("checkUsername: Match found, user check successful.");
				success=true;
				break;
			}
			else
			{
				log("checkUsername: Check failed.");
			}
		}

		return success;
		
		
	}
	
	/**
	 * Converts a Mac adress to a string.
	 *
	 * @param macAddress the mac address
	 * @return the string
	 */
	public static String macAdressToString(byte[] macAddress) {			// found on http://www.koders.com/java/fid95863966C03EAC4031B557B2977907B1F0D1EE66.aspx
		char ch ='-';
		StringBuffer sb = new StringBuffer( 17 );
		for ( int i=44; i>=0; i-=4 ) {
			int nibble =  ((int)( byte2Long(macAddress) >>> i )) & 0xf;
			char nibbleChar = (char)( nibble > 9 ? nibble + ('A'-10) : nibble + '0' );
			sb.append( nibbleChar );
			if ( (i & 0x7) == 0 && i != 0 ) {
				sb.append( ch );
			}
		}
		return sb.toString();		
	}
	
	/**
	 * Converts Byte data to an address in long format.
	 *
	 * @param addr the byte array
	 * @return the address
	 */
	private static long byte2Long(byte addr[]) {
	    long address = 0;
		if (addr != null) {
		    if (addr.length == 6) {
			address = unsignedByteToLong(addr[5]);
			address |= (unsignedByteToLong(addr[4]) << 8);
			address |= (unsignedByteToLong(addr[3]) << 16);
			address |= (unsignedByteToLong(addr[2]) << 24);
			address |= (unsignedByteToLong(addr[1]) << 32);
			address |= (unsignedByteToLong(addr[0]) << 40);
		    } 
		} 
		return address;
	}	
	
	/**
	 * Unsigned byte to long.
	 *
	 * @param b the byte
	 * @return the long value
	 */
	private static long unsignedByteToLong(byte b) {
	    return (long) b & 0xFF;
	}		


	/**
	 * Specifies whether the license was accepted.
	 *
	 * @param licenseAccepted Specifies whether the license was accepted.
	 */
	public void setLicenseAccepted(boolean licenseAccepted) 
	{
		log("setLicenseAccepted: License was accepted by user");
		this.licenseAccepted=licenseAccepted;
	}


	/**
	 * Prints debug information to stdout when printLog is set to true.
	 *
	 * @param message the message
	 */
	private void log(String message)
	{
		if(LicenseChecker.printLog && Main.isLoggingEnabled())
		{
			System.out.println(this.getClass().getName()+"."+message);
		}
	}

}
