/*
 *  Copyright 2010, Enguerrand de Rochefort
 * 
 * This file is part of xdat.
 *
 * xdat is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * xdat is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with xdat.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */


package main;

import java.awt.Color;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;

/**
 * This class stores all user preference settings.
 * 
 * When the user makes changes to settings such as default background color of 
 * charts or delimiting characters for data imports these settings are stored in 
 * an instance of this class. 
 * <p> 
 * Each time a change is made the information is automatically saved by serializing
 * the class instance to a file in the application directory. 
 * <p>
 * User specific settings are supported by naming the settings file after the user
 * name, followed by the extension .pref. 
 * <p>
 * If this file is missing upon startup, a new file with the default settings is created.
 * The default settings used for this purpose are also defined in this class.
 */
public class UserPreferences 
implements Serializable 
{
	
	/** The version tracking unique identifier for Serialization. */
	static final long serialVersionUID = 0002;
	
	/** Flag to enable debug message printing for this class. */
	static final boolean printLog=false;
	//File Options
	/** Open file import browse dialog in the user's home directory by default. 
	 * @see UserPreferences#dirToImportFrom */
	public static final int IMPORT_FROM_HOMEDIR = 0;

	/** Open file import browse dialog in the last opened directory by default. 
	 * @see UserPreferences#dirToImportFrom
	 * @see UserPreferences#lastFile  */
	public static final int IMPORT_FROM_LASTDIR = 1;
	
	/** Open file import browse dialog in a userspecified directory by default. 
	 * @see UserPreferences#dirToImportFrom
	 * @see UserPreferences#userDir  
	 * */
	public static final int IMPORT_FROM_USERDIR = 2;

	/** The directory to import from. 
	 * @see UserPreferences#IMPORT_FROM_HOMEDIR 
	 * @see UserPreferences#IMPORT_FROM_LASTDIR
	 * @see UserPreferences#IMPORT_FROM_USERDIR
	 * @see UserPreferences#lastFile
	 * @see UserPreferences#userDir */
	private int dirToImportFrom = IMPORT_FROM_LASTDIR;
	
	/** Flag to remember when  {@link UserPreferences#lastFile} is initialised. */
	private boolean lastFileInitialised=false;
	
	/** The user's home directory. */
	private String homeDir;
	
	/** The last file that was selected. Defaults to the user's home. */
	private String lastFile= System.getProperty("user.home");
	
	/** User-specified setting where the file browsing dialog should start by default. */
	private String userDir= System.getProperty("user.home");
	
	/** Switch that enables displaying filtered designs. */
	private boolean showFilteredDesigns = false;
	// Chart Display Options
	/** The default background color for charts. */
	private Color backgroundDefaultColor = Color.WHITE;
	//Axis Display Options
	/** The width in pixels that is used by one axis. 
	 * This setting is used to define the axis spacing. The distance of two axes is defined by the sum of their respective widths, divided by two. */
	private int axisWidth=200;
	
	/** The axis height in pixels. */
	private int axisHeight=400;
	
	/** The number of tics used on the axis count. */
	private int axisTicCount=11;
	
	/** The axis color. */
	private Color axisColor = Color.BLACK;
	
	/** The axis font color used for the axis title labels. */
	private Color axisLabelFontColor = Color.BLACK;
	
	/** The font color used for the axis tic labels. */
	private Color axisTicLabelFontColor = Color.BLACK;
	
	/** The axis label font size. */
	private int axisLabelFontSize = 20;
	
	/** The tic label font size. */
	private int ticLabelFontSize = 10;
	
	/** The design label font size. */
	private int designLabelFontSize = 10;
	
	/** The axis tic label format. */
	private String axisTicLabelFormat="%4.3f";
	
	/** The axis tic length in pixels. */
	private int axisTicLength = 4;
	// Axis Filter Display Options
	/** The filter color. */
	private Color filterColor = Color.RED;
	
	/** The height of the triangles that represent the filter in pixels. */
	private int filterHeight = 7;
	
	/** The width of one half triangle that represents a filter in pixels. In other words, 
	 * the filter triangle will be twice as large as the value entered here. */
	private int filterWidth = 5;
	
	/** The active design default color. */
	private Color activeDesignDefaultColor  = Color.GREEN;
	
	/** The inactive (filtered) design default color. Only relevant
	 * when {@link UserPreferences#showFilteredDesigns} is true.*/
	private Color inActiveDesignDefaultColor = Color.GRAY;
	
	/** Flag to enable plotting design id numbers to the left of the left-most axis. */
	private boolean showDesignIDs = true;
	
	/** The design id font size. */
	private int designIDFontSize = 10;
	
	/** The delimiter used to distinguish columns when importing data. */
	private String delimiter ="\\s";
	
	/** Character that is used as delimiter when importing data. Only used to 
	 * store this information internally. When the user makes this setting the character
	 * is also written two the field {@link UserPreferences#delimiter} */
	private String otherDelimiter ="";
	
	/** Flag to enable treating consecutive delimiters as one. */
	private boolean treatConsecutiveAsOne = true;
	
	/** Inverts filters. This means that designs between the filters are shown, while designs
	 * above the top filter and below the bottom filter are not. */
	private boolean filterInverted = false;
	
	/** Flips the axes upside down. */
	private boolean axisInverted = false;
	
	/** Axes are autofitted */
	private boolean autoFitAxis = true;
	
	/** The axis default maximum value. Only relevant if  {@link UserPreferences#autoFitAxis} is false. */
	private double axisDefaultMax=10;
	
	/** The axis default minimum value. Only relevant if  {@link UserPreferences#autoFitAxis} is false. */
	private double axisDefaultMin=-10;

	/**
	 * Gets the axis color.
	 *
	 * @return the axis color
	 */
	public Color getAxisColor() {
		return axisColor;
	}
	
	/**
	 * Sets the axis color.
	 *
	 * @param axisColor the new axis color
	 */
	public void setAxisColor(Color axisColor) {
		this.axisColor = axisColor;
		this.saveToFile();
	}
	
	/**
	 * Gets the axis height.
	 *
	 * @return the axis height
	 */
	public int getAxisHeight() {
		return axisHeight;
	}
	
	/**
	 * Sets the axis height.
	 *
	 * @param axisHeight the new axis height
	 */
	public void setAxisHeight(int axisHeight) {
		this.axisHeight = axisHeight;
		this.saveToFile();
	}
	
	/**
	 * Gets the axis label font color.
	 *
	 * @return the axis label font color
	 */
	public Color getAxisLabelFontColor() {
		return axisLabelFontColor;
	}
	
	/**
	 * Sets the axis label font color.
	 *
	 * @param axisLabelFontColor the new axis label font color
	 */
	public void setAxisLabelFontColor(Color axisLabelFontColor) {
		this.axisLabelFontColor = axisLabelFontColor;
		this.saveToFile();
	}
	
	/**
	 * Gets the axis label font size.
	 *
	 * @return the axis label font size
	 */
	public int getAxisLabelFontSize() {
		return axisLabelFontSize;
	}
	
	/**
	 * Sets the axis label font size.
	 *
	 * @param axisLabelFontSize the new axis label font size
	 */
	public void setAxisLabelFontSize(int axisLabelFontSize) {
		this.axisLabelFontSize = axisLabelFontSize;
		this.saveToFile();
	}
	
	/**
	 * Gets the axis tic count.
	 *
	 * @return the axis tic count
	 */
	public int getAxisTicCount() {
		return axisTicCount;
	}
	
	/**
	 * Sets the axis tic count.
	 *
	 * @param axisTicCount the new axis tic count
	 */
	public void setAxisTicCount(int axisTicCount) {
		this.axisTicCount = axisTicCount;
		this.saveToFile();
	}
	
	/**
	 * Gets the axis tic label font color.
	 *
	 * @return the axis tic label font color
	 */
	public Color getAxisTicLabelFontColor() {
		return axisTicLabelFontColor;
	}
	
	/**
	 * Sets the axis tic label font color.
	 *
	 * @param axisTicLabelFontColor the new axis tic label font color
	 */
	public void setAxisTicLabelFontColor(Color axisTicLabelFontColor) {
		this.axisTicLabelFontColor = axisTicLabelFontColor;
		this.saveToFile();
	}
	
	/**
	 * Gets the axis tic label format.
	 *
	 * @return the axis tic label format
	 */
	public String getAxisTicLabelFormat() {
		return axisTicLabelFormat;
	}
	
	/**
	 * Sets the axis tic label format.
	 *
	 * @param axisTicLabelFormat the new axis tic label format
	 */
	public void setAxisTicLabelFormat(String axisTicLabelFormat) {
		this.axisTicLabelFormat = axisTicLabelFormat;
		this.saveToFile();
	}
	
	/**
	 * Gets the axis tic length.
	 *
	 * @return the axis tic length
	 */
	public int getAxisTicLength() {
		return axisTicLength;
	}
	
	/**
	 * Sets the axis tic length.
	 *
	 * @param axisTicLength the new axis tic length
	 */
	public void setAxisTicLength(int axisTicLength) {
		this.axisTicLength = axisTicLength;
		this.saveToFile();
	}
	
	/**
	 * Gets the axis width.
	 *
	 * @return the axis width
	 */
	public int getAxisWidth() {
		return axisWidth;
	}
	
	/**
	 * Sets the axis width.
	 *
	 * @param axisWidth the new axis width
	 */
	public void setAxisWidth(int axisWidth) {
		this.axisWidth = axisWidth;
		this.saveToFile();
	}
	
	/**
	 * Gets the filter default color.
	 *
	 * @return the filter default color
	 */
	public Color getFilterDefaultColor() {
		return filterColor;
	}
	
	/**
	 * Sets the filter color.
	 *
	 * @param filterColor the new filter color
	 */
	public void setFilterColor(Color filterColor) {
		this.filterColor = filterColor;
		this.saveToFile();
	}
	
	/**
	 * Gets the filter height.
	 *
	 * @return the filter height
	 */
	public int getFilterHeight() {
		return filterHeight;
	}
	
	/**
	 * Sets the filter height.
	 *
	 * @param filterHeight the new filter height
	 */
	public void setFilterHeight(int filterHeight) {
		this.filterHeight = filterHeight;
		this.saveToFile();
	}
	
	/**
	 * Gets the filter width.
	 *
	 * @return the filter width
	 */
	public int getFilterWidth() {
		return filterWidth;
	}
	
	/**
	 * Sets the filter width.
	 *
	 * @param filterWidth the new filter width
	 */
	public void setFilterWidth(int filterWidth) {
		this.filterWidth = filterWidth;
		this.saveToFile();
	}
	
	/**
	 * Gets the axis tic label font size.
	 *
	 * @return the axis tic label font size
	 */
	public int getAxisTicLabelFontSize() {
		return ticLabelFontSize;
	}
	
	/**
	 * Sets the axis tic label font size.
	 *
	 * @param ticLabelFontSize the new axis tic label font size
	 */
	public void setAxisTicLabelFontSize(int ticLabelFontSize) {
		this.ticLabelFontSize = ticLabelFontSize;
		this.saveToFile();
	}
	
	/**
	 * Instantiates a new user preferences.
	 */
	public UserPreferences()
	{
		this.resetToDefault();
		log("Constructor: active design color is now "+this.activeDesignDefaultColor.toString());
	}

	/**
	 * Save to file.
	 */
	public void saveToFile()
   	{	
		try 
		{
	   		FileOutputStream fs = new FileOutputStream(UserPreferences.getPreferenceFileName());
	   		ObjectOutputStream os = new ObjectOutputStream(fs);
	   		os.writeObject(this);
	   		os.close();
   		} catch (IOException e) 
   		{
   			System.err.println(e.toString());
   		}			
   	}
	
	/**
	 * Read from file.
	 *
	 * @param mainWindow the main window
	 * @return the user preferences
	 * @throws InvalidClassException the invalid class exception
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	public static UserPreferences readFromFile(Main mainWindow)
	throws java.io.InvalidClassException, java.io.IOException
   	{
		try {
	   		FileInputStream fs = new FileInputStream(UserPreferences.getPreferenceFileName());
	   		ObjectInputStream is = new ObjectInputStream(fs);
	   		UserPreferences readPrefs = (UserPreferences)is.readObject();
	   		is.close();
	   		return readPrefs;
		} catch (ClassNotFoundException e) {
			System.err.println(e.toString());  	
			return null;
   		}	
   	}	
	
	/**
	 * Gets the design label font size.
	 *
	 * @return the design label font size
	 */
	public int getDesignLabelFontSize() {
		return designLabelFontSize;
	}
	
	/**
	 * Sets the design label font size.
	 *
	 * @param designLabelFontSize the new design label font size
	 */
	public void setDesignLabelFontSize(int designLabelFontSize) {
		this.designLabelFontSize = designLabelFontSize;
	}

	/**
	 * Gets the dir to import from.
	 *
	 * @return the dir to import from
	 */
	public int getDirToImportFrom() {
		return dirToImportFrom;
	}
	
	/**
	 * Sets the dir to import from.
	 *
	 * @param dirToImportFrom the new dir to import from
	 */
	public void setDirToImportFrom(int dirToImportFrom) {
		this.dirToImportFrom = dirToImportFrom;
	}
	
	/**
	 * Gets the last file.
	 *
	 * @return the last file
	 */
	public String getLastFile() 
	{
		return lastFile;
	}
	
	/**
	 * Sets the last file.
	 *
	 * @param lastFileBrowsingDirectory the new last file
	 */
	public void setLastFile(String lastFileBrowsingDirectory) {
		this.lastFile = lastFileBrowsingDirectory;
		this.saveToFile();
	}

	/**
	 * Checks if the last file field has been initialised.
	 *
	 * @return true, if is last file field has been initialised
	 */
	public boolean isLastFileInitialised() 
	{
		return lastFileInitialised;
	}
	
	/**
	 * Gets the user home directory.
	 *
	 * @return the user home directory
	 */
	public String getHomeDir() {
		return homeDir;
	}
	
	/**
	 * Sets the home directory.
	 *
	 * @param homeDir the new home directory
	 */
	public void setHomeDir(String homeDir) {
		this.homeDir = homeDir;
	}
	
	/**
	 * Gets the user directory.
	 *
	 * @return the user directory
	 */
	public String getUserDir() {
		return userDir;
	}
	
	/**
	 * Sets the user directory.
	 *
	 * @param userDir the new user directory
	 */
	public void setUserDir(String userDir) {
		this.userDir = userDir;
	}
	
	/**
	 * Gets the directory for the file browsing dialog.
	 *
	 * @return the current directory based on the user preferences.
	 */
	public String getCurrentDir()
	{
		switch(this.dirToImportFrom)
		{
			case(IMPORT_FROM_HOMEDIR):
			{
				return this.homeDir;
			}
			case(IMPORT_FROM_LASTDIR):
			{
				return this.lastFile;
			}
			case(IMPORT_FROM_USERDIR):
			{
				return this.userDir;
			}
			default:
			{
				return this.homeDir;
			}
		}
		
	}
	

	/**
	 * Gets the delimiter for importing data.
	 *
	 * @return the delimiter
	 */
	public String getDelimiter() {
		return delimiter;
	}
	
	/**
	 * Sets the delimiter for importing data.
	 *
	 * @param delimiter the new delimiter
	 */
	public void setDelimiter(String delimiter) {
		this.delimiter = delimiter;
	}
	
	/**
	 * Checks .
	 *
	 * @return true, if consecutive delimiters should be treated as one
	 */
	public boolean isTreatConsecutiveAsOne() {
		return treatConsecutiveAsOne;
	}
	
	/**
	 * Sets whether consecutive delimiters should be treated as one.
	 *
	 * @param treatConsecutiveAsOne flag for treating consecutive delimiters as one
	 */
	public void setTreatConsecutiveAsOne(boolean treatConsecutiveAsOne) {
		this.treatConsecutiveAsOne = treatConsecutiveAsOne;
	}
	
	/**
	 * Gets the user-defined delimiter.
	 *
	 * @return the user-defined delimiter
	 */
	public String getOtherDelimiter() {
		return otherDelimiter;
	}
	
	/**
	 * Sets the user-defined delimiter.
	 *
	 * @param otherDelimiter the user-defined delimiter
	 */
	public void setOtherDelimiter(String otherDelimiter) {
		this.otherDelimiter = otherDelimiter;
	}
	
	/**
	 * Checks if filtered designs should be shown.
	 *
	 * @return true, if filtered designs are shown
	 */
	public boolean isShowFilteredDesigns() {
		return showFilteredDesigns;
	}
	
	/**
	 * Sets whether filtered designs should be shown.
	 *
	 * @param showFilteredDesigns flag to set whether filtered designs should be shown.
	 */
	public void setShowFilteredDesigns(boolean showFilteredDesigns) {
		this.showFilteredDesigns = showFilteredDesigns;
	}
	
	/**
	 * Gets the active design default color.
	 *
	 * @return the active design default color
	 */
	public Color getActiveDesignDefaultColor() {
		log("getActiveDesignDefaultColor: returning color "+activeDesignDefaultColor.toString());
		return activeDesignDefaultColor;
	}
	
	/**
	 * Sets the active design default color.
	 *
	 * @param activeDesignDefaultColor the new active design default color
	 */
	public void setActiveDesignDefaultColor(Color activeDesignDefaultColor) {
		this.activeDesignDefaultColor = activeDesignDefaultColor;
	}
	
	/**
	 * Gets the filtered design default color.
	 *
	 * @return the filtered design default color
	 */
	public Color getFilteredDesignDefaultColor() {
		return inActiveDesignDefaultColor;
	}
	
	/**
	 * Sets the in active design default color.
	 *
	 * @param inActiveDesignDefaultColor the new in active design default color
	 */
	public void setInActiveDesignDefaultColor(Color inActiveDesignDefaultColor) {
		this.inActiveDesignDefaultColor = inActiveDesignDefaultColor;
	}
	
	/**
	 * Checks if design id numbers are shown left to the left-most axis.
	 *
	 * @return whether design id numbers are shown left to the left-most axis.
	 */
	public boolean isShowDesignIDs() {
		return showDesignIDs;
	}
	
	/**
	 * Specifies whether design id numbers are shown left to the left-most axis.
	 *
	 * @param showDesignIDs specifies whether design id numbers are shown left to the left-most axis.
	 */
	public void setShowDesignIDs(boolean showDesignIDs) {
		this.showDesignIDs = showDesignIDs;
	}
	
	/**
	 * Gets the design id font size.
	 *
	 * @return the design id font size
	 */
	public int getDesignIDFontSize() {
		return designIDFontSize;
	}
	
	/**
	 * Sets the design id font size.
	 *
	 * @param designIDFontSize the new design id font size
	 */
	public void setDesignIDFontSize(int designIDFontSize) {
		this.designIDFontSize = designIDFontSize;
	}

	/**
	 * Gets the default background color.
	 *
	 * @return the default background color
	 */
	public Color getDefaultBackgroundColor() {
		return backgroundDefaultColor;
	}
	
	/**
	 * Sets the default background color.
	 *
	 * @param backgroundColor the new default background color
	 */
	public void setDefaultBackgroundColor(Color backgroundColor) {
		this.backgroundDefaultColor = backgroundColor;
	}
	
	/**
	 * Checks whether filters are inverted.
	 *
	 * @return true, if filters are inverted.
	 */
	public boolean isFilterInverted() {
		return filterInverted;
	}
	
	/**
	 * Specifies whether filters inverted.
	 *
	 * @param filterInverted Specifies whether filters inverted.
	 */
	public void setFilterInverted(boolean filterInverted) {
		this.filterInverted = filterInverted;
	}
	
	/**
	 * Prints debug information to stdout when printLog is set to true.
	 *
	 * @param message the message
	 */
	private void log(String message)
	{
		if(UserPreferences.printLog && Main.isLoggingEnabled())
		{
			System.out.println(this.getClass().getName()+"."+message);
		}
	}
	
	/**
	 * Checks whether axes are inverted.
	 *
	 * @return true, if axes are inverted
	 */
	public boolean isAxisInverted() {
		return axisInverted;
	}
	
	/**
	 * Specifies whether axes are inverted.
	 *
	 * @param axisInverted Specifies whether axes are inverted.
	 */
	public void setAxisInverted(boolean axisInverted) {
		log("setAxisInverted called with parameter "+axisInverted);
		this.axisInverted = axisInverted;
	}
	
	/**
	 * Checks if axes are autofitted.
	 *
	 * @return true, if axes are autofitted.
	 */
	public boolean isAutoFitAxis() {
		return autoFitAxis;
	}
	
	/**
	 * Specifies whether axes should be autofitted.
	 *
	 * @param autoFitAxis Specifies whether axes should be autofitted.
	 */
	public void setAutoFitAxis(boolean autoFitAxis) {
		this.autoFitAxis = autoFitAxis;
	}
	
	/**
	 * Gets the axis default maximum value.
	 *
	 * @return the axis default maximum value
	 */
	public double getAxisDefaultMax() {
		return axisDefaultMax;
	}
	
	/**
	 * Gets the axis default minimum value.
	 *
	 * @return the axis default minimum value
	 */
	public double getAxisDefaultMin() {
		return axisDefaultMin;
	}
	
	/**
	 * Sets the axis default minimum value.
	 *
	 * @param axisDefaultMin the new axis default minimum value
	 */
	public void setAxisDefaultMin(double axisDefaultMin) {
		this.axisDefaultMin = axisDefaultMin;
	}
	
	/**
	 * Sets the axis default max.
	 *
	 * @param axisDefaultMax the new axis default max
	 */
	public void setAxisDefaultMax(double axisDefaultMax) {
		this.axisDefaultMax = axisDefaultMax;
	}
	
	/**
	 * Gets the preference file name.
	 *
	 * @return the preference file name
	 */
	public static String getPreferenceFileName()
	{
		return System.getProperty("user.name")+".pref";
	}
	
	/**
	 * Restores all default settings.
	 */
	public void resetToDefault()
	{
		this.lastFileInitialised=false;
		this.dirToImportFrom = IMPORT_FROM_LASTDIR;
		this.homeDir = System.getProperty("user.home");
		this.lastFile= System.getProperty("user.home");
		this.userDir= System.getProperty("user.home");
		this.showFilteredDesigns = false;
		// Chart Display Options
		this.backgroundDefaultColor = Color.WHITE;
		//Axis Display Options
		this.axisWidth=200;
		this.axisHeight=400;
		this.axisTicCount=11;
		this.axisColor = Color.BLACK;
		this.axisLabelFontColor = Color.BLACK;
		this.axisTicLabelFontColor = Color.BLACK;
		this.axisLabelFontSize = 20;
		this.ticLabelFontSize = 10;
		this.designLabelFontSize = 10;
		this.axisTicLabelFormat="%4.3f";
		this.axisTicLength = 4;
		// Axis Filter Display Options
		this.filterColor = Color.RED;
		this.filterHeight = 7;
		this.filterWidth = 5;
		this.activeDesignDefaultColor  = Color.GREEN;
		this.inActiveDesignDefaultColor = Color.GRAY;
		this.showDesignIDs = true;
		this.designIDFontSize = 10;
		this.delimiter ="\\s";
		this.otherDelimiter ="";
		this.treatConsecutiveAsOne = true;
		this.filterInverted = false;
		this.axisInverted = false;
		this.autoFitAxis = true;
		this.axisDefaultMax=10;
		this.axisDefaultMin=-10;		
	}

}
