/*
 *  Copyright 2010, Enguerrand de Rochefort
 * 
 * This file is part of xdat.
 *
 * xdat is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * xdat is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with xdat.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package data;

import java.io.Serializable;
import java.util.*;

import main.Main;
import exceptions.CorruptDataException;

/**
 * A Parameter represents a parameter of a {@link data.DataSheet}.
 * <p>
 * It is used to store information about the type of data stored in a column
 * of the DataSheet. 
 * <p>
 * Two data types are supported: Numeric and Discrete.
 * <p>
 * Numeric parameters are used for columns that only contain numbers.
 * Discrete parameters are used for all columns that contain at least one
 * non-numeric value. All values are stored in a TreeSet and sorted in
 * alphabetical order. This makes it possible to also treat information on
 * parameters that are not quantifiable, such as different shapes of an object
 * or similar.
 * 
 */
public class Parameter 
implements Serializable
{
	
	/** The version tracking unique identifier for Serialization. */
	static final long serialVersionUID = 0002;
	
	/** Flag to enable debug message printing for this class. */
	static final boolean printLog=false;
	
	/** The parameter name. */
	private String name;
	
	/** Specifies whether the parameter is numeric. If it is not, it is discrete. */
	private boolean numeric = true;
	
	/** The discrete levels. Only applies for non-numeric parameters. */
	private TreeSet<String> discreteLevels = new TreeSet<String>(new ReverseStringComparator());
	
	/**
	 * Instantiates a new parameter.
	 *
	 * @param name the parameter name
	 */
	public Parameter (String name)
	{
		this.name = name;
	}
	
	/**
	 * Gets the name.
	 *
	 * @return the name
	 */
	public String getName() {
		return name;
	}
	
	/**
	 * Sets the name.
	 *
	 * @param name the new name
	 */
	public void setName(String name) {
		this.name = name;
	}
	
	/**
	 * Checks whether the parameter is numeric or discrete.
	 *
	 * @return true, if the parameter is numeric
	 */
	public boolean isNumeric() {
		return numeric;
	}
	
	/**
	 * Specifies whether the parameter is numeric or dicrete.
	 *
	 * @param numeric specifies if the parameter is numeric
	 */
	public void setNumeric(boolean numeric) {
		this.numeric = numeric;
	}
	
	/**
	 * Gets a numeric representation of a string value for this parameter.
	 * <p>
	 * If the parameter is numeric, an attempt is made to parse the string as a Double.
	 * If this attempt leads to a NumberFormatException, the parameter is not considered
	 * numeric anymore, but is transformed into a discrete parameter.
	 * <p>
	 * If the parameter is not numeric, the string is looked up in the TreeSet discreteLevels
	 * that should contain all discrete values (that is Strings) that were found in the data
	 * sheet for this parameter. If the value is not found it is added as a new discrete level
	 * for this parameter. The treeSet is then searched again in order to get the correct
	 * index of the new discrete level. 
	 * <p>
	 * If this second search does not yield the result, something unexpected has gone wrong and 
	 * a CorruptDataException is thrown. 
	 *
	 * @param string the string
	 * @return the numeric representation of the given string
	 */
	public double getDoubleValueOf(String string)
	{
		if (this.numeric)
		{
			try {
				return Double.parseDouble(string);
			} catch (NumberFormatException e) {
				this.setNumeric(false);
			}
		}
		int index=0;
		Iterator<String> it = discreteLevels.iterator();
		log("getDoubleValueOf: checking index of string "+string);
		while(it.hasNext())
		{
			if(string.equalsIgnoreCase(it.next()))
			{
				return (double)index;
			}
			index++;
		}
		
		// String not found, add it to discrete levels
		log("getDoubleValueOf: string "+string+" not found, adding it to tree set ");
		this.discreteLevels.add(string);
		index=0;
		it = discreteLevels.iterator();
		log("getDoubleValueOf: re-checking index of string "+string);
		while(it.hasNext())
		{
			if(string.equalsIgnoreCase(it.next()))
			{
				return (double)index;
			}
			index++;
		}
		throw new CorruptDataException(this);	
	}
	
	/**
	 * Gets the string representation of a given double value for this parameter.
	 * <p>
	 * If the parameter numeric, the provided value is simply converted to a String and returned.
	 * <p>
	 * If it is discrete, the double value is casted to an Integer value and this value is used
	 * as an index to look up the corresponding discrete value string in the TreeSet discreteLevels.
	 * <p>
	 * If no value is found for the given index the data is assumed to be corrupt and a CorruptDataException
	 * is thrown.
	 * 
	 * @param value the numeric value
	 * @return the string representation of the given double value for this parameter.
	 */
	public String getStringValueOf(double value)
	{
		if (this.numeric)
		{
			log("getStringValueOf: Parameter "+this.name+" is numeric. Returning "+(Double.toString(value)));
			return Double.toString(value);
		}
		else
		{
			log("getStringValueOf: Parameter "+this.name+" is not numeric. ");
			log("value = "+value);
			int index = (int)value;
			log("getStringValueOf: index is "+index);
			int currentIndex = 0;
			Iterator<String> it = discreteLevels.iterator();
			while (it.hasNext())
			{
				log("getStringValueOf: checking currentIndex "+currentIndex+" against index "+index);
				String next = it.next();
				if(currentIndex == index)
				{
					log("getStringValueOf: check positive. Returning string "+next);
					return next;
				}
				currentIndex++;
			}
			throw new CorruptDataException(this);
		}
	}

	/**
	 * Gets the discrete level count in the TreeSet that stores all discrete levels.
	 * <p>
	 * Only applies to non-numeric parameters.
	 *
	 * @return the discrete level count
	 */
	public int getDiscreteLevelCount()
	{
		if (this.isNumeric())
		{
			throw new RuntimeException("Parameter "+this.name+" is numeric!");
		}
		else
		{
//			log("getDiscreteLevelCount returning size "+this.discreteLevels.size());
			return this.discreteLevels.size();
		}
	}

	/**
	 * Reset discrete levels to an empty TreeSet.
	 */
	public void resetDiscreteLevels()
	{
		log("resetDiscreteLevels called");
		this.discreteLevels = new TreeSet<String>(new ReverseStringComparator());
	}
	
	/**
	 * Prints debug information to stdout when printLog is set to true.
	 *
	 * @param message the message
	 */
	private void log(String message)
	{
		if(Parameter.printLog && Main.isLoggingEnabled())
		{
			System.out.println(this.getClass().getName()+"."+message);
		}
	}
		
	/**
	 * Comparator for the discrete levels TreeSet to sort the data alphabetically.
	 */
	class ReverseStringComparator
	implements Comparator<String>, Serializable
	{
		
		/** The version tracking unique identifier for Serialization. */
		static final long serialVersionUID = 0000;
		
		/* (non-Javadoc)
		 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
		 */
		public int compare(String s1, String s2)
		{
			return (s2.compareToIgnoreCase(s1));
		}
	}	
	
}
