/*
 *  Copyright 2010, Enguerrand de Rochefort
 * 
 * This file is part of xdat.
 *
 * xdat is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * xdat is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with xdat.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package data;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.io.Serializable;
import java.util.Vector;
import javax.swing.event.*;
import javax.swing.table.*;
import chart.Chart;
import exceptions.InconsistentDataException;
import main.Main;
import main.UserPreferences;

/**
 * A representation of the data imported from a text file.
 * <p>
 * Everytime the user imports data from a text file the data is stored in
 * a DataSheet. The data sheet is kind of wrapper class for the collection 
 * of all rows in the text file. Each row represents one {@link data.Design}.
 * <p>
 * In addition to storing the Designs and providing the possibility to display
 * them in a JTable by implementing TableModel, the DataSheet class also 
 * keeps track of the Parameters in the data set. Each column represents one 
 * {@link data.Parameter}.
 * <p> 
 * The third main function of this class is to actually read the data from a 
 * text file. While doing this, the DataSheet also collects some additional information, 
 * such as 
 * <ul>
 * <li>the parameter types (numeric/discrete, see the Parameter class for further info)
 * <li>the Parameter names. These are obtained from a header line in the data or are given
 * default names such as Parameter 1, Parameter 2 and so on.
 * </ul>
 * Note that floating point values must be formatted in american format. (#.##) instead of (#,##)
 * <p>
 * Finally, the DataSheet also keeps track of all {@link data.Cluster}s. However, it does not
 * store the information to which Cluster each Design belongs. This information is stored in 
 * the Designs themselves. It is important to understand this because it means that whenever
 * {@link DataSheet#updateData(String, boolean)} is called this information is lost.
 */
public class DataSheet 
implements TableModel, Serializable 
{
	
	/** The version tracking unique identifier for Serialization. */
	static final long serialVersionUID = 0006;
	
	/** Flag to enable debug message printing for this class. */
	static final boolean printLog=false;
	
	/** The user preferences. */
	private UserPreferences userPreferences;
	
	/** The cluster set. */
	private ClusterSet clusterSet;
	
	/** The Vector containing all Designs. */
	private Vector<Design> data = new Vector<Design>(0,1);
	
	/** The parameters. */
	private Vector<Parameter> parameters = new Vector<Parameter>(0,1);
	
	/** Table Model Listeners to enable updating the GUI. */
	private transient Vector <TableModelListener>listeners = new Vector<TableModelListener>();
	
	/** The delimiter used for the data import. */
	private String delimiter;

	/**
	 * Instantiates a new data sheet and fills it with data from a given file.
	 * <p>
	 * Uses {@link #importData(String, boolean)} for the step of reading the data.
	 *
	 * @param pathToInputFile the path to the input file
	 * @param userPreferences the user preferences
	 * @param dataHasHeaders specifies whether the data has headers to read the Parameter names from.
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	public DataSheet(String pathToInputFile, UserPreferences userPreferences, boolean dataHasHeaders)
	throws IOException
	{
		this.userPreferences = userPreferences;
		this.clusterSet = new ClusterSet(this.userPreferences, this);
		this.delimiter=this.userPreferences.getDelimiter();
		if(this.userPreferences.isTreatConsecutiveAsOne())
			this.delimiter = this.delimiter + "+";
		importData(pathToInputFile, dataHasHeaders);

	}

	/**
	 * Scans the whole DataSheet to determine the parameter types (numeric/discrete). 
	 * @see data.Parameter
	 */
	private void checkParameters() 
	{
		for(int j=0; j<this.parameters.size();j++)
		{
			this.parameters.get(j).resetDiscreteLevels();
			for (int i=0; i<this.data.size(); i++)
			{
				log("checkParameters: checking parameter "+this.parameters.get(j).getName());
				log("checkParameters: parameter.isNumeric() before check: "+this.parameters.get(j).isNumeric());
				data.get(i).getDoubleValue(this.parameters.get(j));			// getDoubleValue sets parameter to non-numeric if value cannot be parsed to double
				log("checkParameters: parameter.isNumeric() after check: "+this.parameters.get(j).isNumeric());
			}
		}
	}	

	/**
	 * Fills the DataSheet with data from a given file and assigns Parameter names.
	 * <p>
	 * Note that floating point values must be formatted in american format. (#.##) instead of (#,##)
	 * <p>
	 * If dataHasHeaders is true, the Parameter names are read from the first line. Otherwise
	 * the parameter names are created automatically.
	 *
	 * @param pathToInputFile the path to the input file
	 * @param dataHasHeaders specifies whether the data has headers to read the Parameter names from.
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	private void importData(String pathToInputFile, boolean dataHasHeaders) 
	throws IOException
	{
		BufferedReader f;	
		String line;

		int idCounter=1;
		f = new BufferedReader(
		new FileReader(pathToInputFile));
		line = (f.readLine()).trim();
		
		String[] lineElements = line.split(this.delimiter);
//		log("importData: dataHasHeaders is "+dataHasHeaders);
		if(dataHasHeaders)	// if data has headers read the parameter names from the first line
		{
			for (int i=0; i<lineElements.length; i++)
			{
//				log("importData: adding parameter "+lineElements[i]);
				this.parameters.add(new Parameter(this.getUniqueParameterName(lineElements[i])));
			}	
		}
		else  // if data does not have headers read the first Design from the first line and create default Parameter names
		{
			Design newDesign = new Design(idCounter++);
			for (int i=0; i<lineElements.length; i++)
			{
//				log("importData: adding parameter "+("Parameter "+(i+1)));
				this.parameters.add(new Parameter("Parameter "+(i+1)));
				newDesign.setValue(this.parameters.get(i), lineElements[i]);
//				log("importData: design ID "+newDesign.getId()+": setting value of parameter "+("Parameter "+(i+1))+" to "+lineElements[i]);

			}
			this.data.add(newDesign);
		}			
		while ((line = f.readLine()) != null)  // read all subsequent lines into Designs
		{
			Design newDesign;
			lineElements = line.split(this.delimiter);
			if(lineElements.length<1)
			{
				// skip line
			}
			else
			{
				newDesign = new Design(idCounter++);
				boolean newDesignContainsValues = false;	// flag to prevent reading in empty lines
				for (int i=0; i<lineElements.length; i++)
				{
					if(lineElements[i].length()>0 && (!lineElements[i].equals(new String("\\s"))))
					{
						newDesignContainsValues = true;		// found non-empty empty field on the line
					}
					
				}				
				if(newDesignContainsValues)			// only continue if at least one non-empty field was found on the line
				{
					for (int i=0; i<this.parameters.size(); i++)
					{
						if(lineElements.length<=i || lineElements[i].length()<=0 || lineElements[i].equals(new String("\\s")))
						{
							newDesign.setValue(this.parameters.get(i), "-");
						}
						else
						{
							newDesignContainsValues = true;		// found a non-empty field on the line
							newDesign.setValue(this.parameters.get(i), lineElements[i]);
						}
	//					log("importData: design ID "+newDesign.getId()+": setting value of parameter "+this.parameters.get(i).getName()+" to "+newDesign.getStringValue(this.parameters.get(i)));
						
					}
					this.data.add(newDesign);
				}
			}
		}
		f.close();
		checkParameters();
	}

	/**
	 * Updates the DataSheet with Data from a given file and assigns Parameter names. 
	 * <p>
	 * If dataHasHeaders is true, the Parameter names are read from the first line. Otherwise
	 * the parameter names are created automatically.
	 * <p>
	 * The difference of updating vs. importing is that all Charts are kept. This requires the new data
	 * to have the same number of parameters as the previous one. Otherwise the InconsistentDataException
	 * is thrown. 
	 * <p>
	 * @param pathToInputFile the path to the input file
	 * @param dataHasHeaders specifies whether the data has headers to read the Parameter names from.
	 * @throws IOException Signals that an I/O exception has occurred.
	 * @throws InconsistentDataException if the user tries to import data that does not have the same number
	 * of columns as the current DataSheet.
	 */
	public void updateData(String pathToInputFile, boolean dataHasHeaders) 
	throws IOException, InconsistentDataException
	{
		BufferedReader f;	
		String line;

		int idCounter=1;
		f = new BufferedReader(
		new FileReader(pathToInputFile));
		line = (f.readLine()).trim();
		
		String[] lineElements = line.split(this.delimiter);
		
		if(lineElements.length != this.getParameterCount())
			throw new InconsistentDataException(pathToInputFile);
		
		this.data.removeAllElements();

//		log("updateData: data has "+data.size()+" items.");


		if(dataHasHeaders)	// if data has headers read the parameter names from the first line
		{		
			for (int i=0; i<this.parameters.size(); i++)
			{
				this.parameters.get(i).setName(null);
			}	
			for (int i=0; i<lineElements.length; i++)
			{
				this.parameters.get(i).setName(this.getUniqueParameterName(lineElements[i]));
//				log("updateData: parameter name set to "+this.parameters.get(i).getName());
			}	
		}	
		else   // if data does not have headers read the first Design from the first line and create default Parameter names
		{
			Design newDesign = new Design(idCounter++);				
			for (int i=0; i<this.parameters.size(); i++)
			{
				if(lineElements.length<=i)
				{
					newDesign.setValue(this.parameters.get(i), "-");
				}
				else
				{
					newDesign.setValue(this.parameters.get(i), lineElements[i]);
				}
			
			}
			this.data.add(newDesign);
			{
				this.data.add(newDesign);
			}
		}
		while ((line = f.readLine()) != null)    // read all subsequent lines into Designs
		{
			Design newDesign;
			lineElements = line.split(this.delimiter);
			if(lineElements.length<1)
			{
				// skip line
			}
			else
			{
				newDesign = new Design(idCounter++);
				boolean newDesignContainsValues = false;	// flag to prevent reading in empty lines
				for (int i=0; i<lineElements.length; i++)
				{
					if(lineElements[i].length()>0 && (!lineElements[i].equals(new String("\\s"))))
					{
						newDesignContainsValues = true;		// found non-empty empty field on the line
					}
					
				}

				if(newDesignContainsValues)			// only continue if at least one non-empty field was found on the line
				{
					for (int i=0; i<this.parameters.size(); i++)
					{
						if(lineElements.length<=i || lineElements[i].length()<=0 || lineElements[i].equals(new String("\\s")))
						{
							newDesign.setValue(this.parameters.get(i), "-");
						}
						else
						{
							newDesign.setValue(this.parameters.get(i), lineElements[i]);
							newDesignContainsValues = true;		// found a non-empty field on the line
						}
					
					}
					this.data.add(newDesign);
				}
			}
		}
		f.close();
		checkParameters();
		
	}
	
	
	/* (non-Javadoc)
	 * @see javax.swing.table.TableModel#getColumnClass(int)
	 */
	public Class<?> getColumnClass(int arg0) {
		return String.class;
	}

	/* (non-Javadoc)
	 * @see javax.swing.table.TableModel#getColumnCount()
	 */
	public int getColumnCount() 
	{
		return parameters.size();
	}
	
	/* (non-Javadoc)
	 * @see javax.swing.table.TableModel#getRowCount()
	 */
	public int getRowCount() {
		return data.size();
	}
	
	/* (non-Javadoc)
	 * @see javax.swing.table.TableModel#getColumnName(int)
	 */
	public String getColumnName(int columnIndex) {
		return parameters.get(columnIndex).getName();
	}

	/* (non-Javadoc)
	 * @see javax.swing.table.TableModel#getValueAt(int, int)
	 */
	public Object getValueAt(int rowIndex, int columnIndex) 
	{
		if(columnIndex == 0)
		{
			return this.data.get(rowIndex).getId();
		}
		else
		{
			return this.data.get(rowIndex).getStringValue(parameters.get(columnIndex-1));
		}
	}

	/* (non-Javadoc)
	 * @see javax.swing.table.TableModel#setValueAt(java.lang.Object, int, int)
	 */
	public void setValueAt(Object arg0, int rowIndex, int columnIndex) 
	{
		try {
			this.data.get(rowIndex).setValue(parameters.get(columnIndex-1), arg0.toString());
			Double.parseDouble(arg0.toString());
		} catch (NumberFormatException e) {
			log("setValueAt: value "+arg0.toString()+" of parameter "+this.parameters.get(columnIndex-1).getName()+" is not numeric.");
			this.parameters.get(columnIndex-1).setNumeric(false);
			log("setValueAt: value "+arg0.toString()+" is not numeric.");
			this.checkParameters();
		}

	}
	
	/* (non-Javadoc)
	 * @see javax.swing.table.TableModel#isCellEditable(int, int)
	 */
	public boolean isCellEditable(int rowIndex, int columnIndex) 
	{
		if(columnIndex == 0)
		{
			return false;
		}
		else
		{
			return true;
		}
	}

	/* (non-Javadoc)
	 * @see javax.swing.table.TableModel#addTableModelListener(javax.swing.event.TableModelListener)
	 */
	public void addTableModelListener(TableModelListener l) 
	{
		if(listeners == null)
			listeners = new Vector<TableModelListener>();
		listeners.add(l); 
	} 
	
	/* (non-Javadoc)
	 * @see javax.swing.table.TableModel#removeTableModelListener(javax.swing.event.TableModelListener)
	 */
	public void removeTableModelListener(TableModelListener l)
	{
		listeners.remove(l); 
	}

	/**
	 * Fire table changed.
	 */
	public void fireTableChanged()
	{
		TableModelEvent e = new TableModelEvent( this);
		for( int i = 0, n = listeners.size(); i<n; i++ )
		{ 
			((TableModelListener)listeners.get(i)).tableChanged(e); 
		}
	}
	
	/**
	 * Gets the Design with index i.
	 *
	 * @param i the index
	 * @return the Design
	 */
	public Design getDesign(int i)
	{
		return this.data.get(i);
	}
	
	/**
	 * Adds a Design to the DataSheet.
	 *
	 * @param design the Design
	 */
	public void addDesign(Design design)
	{
		this.data.add(design);
		fireTableChanged();
	}

	/**
	 * Removes a Design from the DataSheets.
	 *
	 * @param design the Design
	 */
	public void removeDesign(Design design)
	{
		this.data.remove(design);
		fireTableChanged();
	}

	/**
	 * Gets the Pparameter count.
	 *
	 * @return the Parameter count
	 */
	public int getParameterCount()
	{
		return this.parameters.size();
	}
	
	/**
	 * Gets the Parameter name of the Parameter with index index.
	 *
	 * @param index the index
	 * @return the parameter name
	 */
	public String getParameterName(int index)
	{
		if(index>=this.parameters.size() || index <0)
			throw new IllegalArgumentException("Invalid Index "+index);
		return this.parameters.get(index).getName();
	}

	/**
	 * Gets the Parameter with the index index.
	 *
	 * @param index the Parameter index
	 * @return the Parameter
	 */
	public Parameter getParameter(int index)
	{
		if(index>=this.parameters.size() || index <0)
			throw new IllegalArgumentException("Invalid Index "+index);
		return this.parameters.get(index);
	}
	
	/**
	 * Gets the Parameter with the name parameterName.
	 *
	 * @param parameterName the Parameter name
	 * @return the Parameter
	 */
	public Parameter getParameter(String parameterName)
	{
		for(int i=0; i<this.parameters.size(); i++)
		{
			if(parameterName.equals(this.parameters.get(i).getName()))
			{
				return this.parameters.get(i);
			}
		}
		throw new IllegalArgumentException("Parameter "+parameterName+" not found");
	}
	
	/**
	 * Gets the maximum value of a given Parameter in the DataSheet.
	 *
	 * @param param the Parameter
	 * @return the maximum value of the given Parameter.
	 */
	public double getMaxValueOf(Parameter param)
	{
		if (param.isNumeric())
		{
			double max = Double.NEGATIVE_INFINITY;
			for (int i=0; i<this.data.size();i++)
			{
				log("getMaxValueOf: max = "+max);
				if(max<this.data.get(i).getDoubleValue(param))
				{
	
					log("getMaxValueOf: Higher value found. New max = "+max);
					max = this.data.get(i).getDoubleValue(param);
				}
			}
	
			log("getMaxValueOf: Final max = "+max);
			return max;
		}
		else
		{
			return param.getDiscreteLevelCount()-1;
		}
	}
	
	/**
	 * Gets the minimum value of a given Parameter.
	 *
	 * @param param the parameter
	 * @return the minimum value of the given Parameter.
	 */
	public double getMinValueOf(Parameter param)
	{
		if (param.isNumeric())
		{	
			double min = Double.POSITIVE_INFINITY;
			for (int i=0; i<this.data.size();i++)
			{
				if(min>this.data.get(i).getDoubleValue(param))
				{
					min = this.data.get(i).getDoubleValue(param);
				}
			}
			return min;
		}
		else
		{
			return 0.0;
		}
	}
	
	/**
	 * Gets the design count.
	 *
	 * @return the design count
	 */
	public int getDesignCount()
	{
		return this.data.size();
	}
	
	/**
	 * Checks whether a given String is a unique parameter name using {@link #isNameUnique(String)}. Returns a unique
	 * name if the provided string is not unique and the unchanged String otherwise. 
	 *
	 * @param nameSuggestion the name that should be checked
	 * @return the a unique name if nameSuggestion was not a unique name and nameSuggestion otherwise
	 */
	private String getUniqueParameterName(String nameSuggestion)
	{
		String name = nameSuggestion;
		int id = 2;
		while(!isNameUnique(name))
			name = nameSuggestion +" ("+(id++)+")";			
//		log("getUniqueParameterName: returning name "+name);
		return name;
	}
	
	/**
	 * Checks if a give String is a unique Parameter name.
	 *
	 * @param name the name to check
	 * @return true, if the name is unique
	 */
	private boolean isNameUnique(String name)
	{
		boolean unique = true;
		for (int i = 0; i<this.parameters.size(); i++)
		{
			if(name.equals(this.parameters.get(i).getName()))
			{
				unique = false;
				break;
			}
		}
		return unique;
	}	
	
	/**
	 * Prints debug information to stdout when printLog is set to true.
	 *
	 * @param message the message
	 */
	private void log(String message)
	{
		if(DataSheet.printLog && Main.isLoggingEnabled())
		{
			System.out.println(this.getClass().getName()+"."+message);
		}
	}

	/**
	 * Gets the user preferences.
	 *
	 * @return the user preferences
	 */
	public UserPreferences getUserPreferences() {
		return userPreferences;
	}

	/**
	 * Gets the cluster set.
	 *
	 * @return the cluster set
	 */
	public ClusterSet getClusterSet() {
		return clusterSet;
	}

	/**
	 * Sets the cluster set.
	 *
	 * @param clusterSet the new cluster set
	 */
	public void setClusterSet(ClusterSet clusterSet) {
		this.clusterSet = clusterSet;
	}

	/**
	 * Evaluate each Design to check whether it is within all axis bounds. 
	 *
	 * @param chart the chart
	 * @see Design
	 */
	public void evaluateBoundsForAllDesigns(Chart chart)
	{
		for(int i = 0; i<this.getDesignCount(); i++)
		{
			this.data.get(i).evaluateBounds(chart);
		}
	}


}
